/**
 *  Copyright 2002-2006 Peter Seiderer <Peter.Seiderer@ciselant.de>
 *
 *  This file is part of SeBIE.
 *
 *  SeBIE is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  SeBIE is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with SeBIE; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
#include "sebie_callbacks.h"
#include "sebie_helpers.h"
#include "sebie_gtk_helpers.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#define SEBIE_CATCH_PIXEL_COUNT  15


void
on_treeview_file_list_changed(GtkTreeSelection *selection,
			      gpointer user_data)
{
  struct sebie_s *sebie = (struct sebie_s *)user_data;
  GtkTreeIter iter;
  GtkTreeModel *model;
  gchar *filename;
  if (gtk_tree_selection_get_selected(selection, &model, &iter)) {
    gtk_tree_model_get(model, &iter, 0, &filename, -1);
    sebie_set_string(&sebie->model.input_file_name, filename);
    load_pixbuf(sebie);
    int nmatch = 2;
    regmatch_t matchptr[nmatch];
    if (regexec(&sebie->model.regex, sebie->model.input_file_name, nmatch, matchptr, 0) == 0) {
      char buf_regex[nmatch][256];
      int i;
      for (i = 0; i < nmatch; i++) {
	strncpy(buf_regex[i],sebie->model.input_file_name +  matchptr[i].rm_so, matchptr[i].rm_eo - matchptr[i].rm_so);
	buf_regex[i][matchptr[i].rm_eo - matchptr[i].rm_so] = '\0';
	sebie_debug("buf_regex[%d] = %s\n", i, buf_regex[i]);
      }
      char buf[256];
      snprintf(buf, 256, sebie->model.output_format, buf_regex[1]);
      sebie_set_string(&sebie->model.output_file_name, buf);
      sebie->dirty_tags |= SEBIE_UPDATE_OUTPUT_FILE_NAME;
      sebie_update_view(sebie);
    }
  }
}

void
on_entry_input_path_changed (GtkEditable *editable,
			     gpointer    user_data)
{
  struct sebie_s *sebie = (struct sebie_s *)user_data;
  G_CONST_RETURN gchar *input_path = gtk_entry_get_text(GTK_ENTRY(editable));
  sebie_set_string(&sebie->model.input_path, input_path);
  sebie->model_tags |= SEBIE_MODEL_INPUT_PATH;
  sebie_update_file_list(sebie);
}


void
on_entry_input_regex_changed (GtkEditable *editable,
			     gpointer    user_data)
{
  struct sebie_s *sebie = (struct sebie_s *)user_data;
  G_CONST_RETURN gchar *input_regex = gtk_entry_get_text(GTK_ENTRY(editable));
  sebie_debug("input_regex = %s\n", input_regex);
  sebie_set_string(&sebie->model.input_regex, input_regex);
  sebie->model_tags |= SEBIE_MODEL_INPUT_REGEX;
  int err;
  if ((err = regcomp(&sebie->model.regex, input_regex, 0)) != 0) {
    char buf[256];
    regerror(err, &sebie->model.regex, buf, 256);
    fprintf(stderr, "Error: regcomp failed (%s)\n", buf);
    return;
  }
  sebie_update_file_list(sebie);
}

void
on_entry_output_format_changed (GtkEditable *editable,
				gpointer    user_data)
{
  struct sebie_s *sebie = (struct sebie_s *)user_data;
  G_CONST_RETURN gchar *output_format = gtk_entry_get_text(GTK_ENTRY(editable));
  sebie_debug("output_format = %s\n", output_format);
  sebie_set_string(&sebie->model.output_format, output_format);
  sebie->model_tags |= SEBIE_MODEL_OUTPUT_FORMAT;
  save_sebie_config(sebie);
}


gint
window1_delete_event (GtkWidget *widget,
		      GdkEvent  *event,
		      gpointer   data)
{
    sebie_debug("window1_delete_event\n");
    return FALSE;
}


void
window1_destroy (GtkWidget *widget,
		 gpointer   data)
{
    sebie_debug("window1_destroy\n");
    gtk_main_quit ();
}


void
drawingarea1_size_allocate (GtkWidget *widget,
			    GtkAllocation *allocation,
			    gpointer   user_data)
{
    struct sebie_s *sebie = (struct sebie_s *)user_data;
    gint window1_width;
    gint window1_height;

    sebie_debug("drawingarea1_size_allocate x = %d y = %d width = %d height = %d\n",
		allocation->x,
		allocation->y,
		allocation->width,
		allocation->height);
 
    gtk_window_get_size(sebie->view.window1, &window1_width, &window1_height);
    sebie->model.window1_width = window1_width;
    sebie->model.window1_height = window1_height;
    sebie->model.pixbuf_width = allocation->width;
    sebie->model.pixbuf_height = allocation->height;
 
    sebie->model_tags |= SEBIE_MODEL_CHANGE_PIXBUF_SIZE;
    save_sebie_config(sebie);
}


void
on_button_quit_clicked (GtkButton *button,
			gpointer  user_data)
{
  gtk_main_quit ();
}


void
on_button_save_clicked (GtkButton *button,
			gpointer  user_data)
{
  struct sebie_s *sebie = (struct sebie_s *)user_data;

  sebie_debug("image_width:  %d\n", sebie->model.image_width);
  sebie_debug("image_height: %d\n", sebie->model.image_height);
  sebie_debug("image_gamma:  %lf\n", sebie->model.image_gamma);
  sebie_debug("selection_x:  %d\n", sebie->model.selection_x);
  sebie_debug("selection_y:  %d\n", sebie->model.selection_y);
  sebie_debug("selection_width:  %d\n", sebie->model.selection_width);
  sebie_debug("selection_height: %d\n", sebie->model.selection_height);
  sebie_debug("scaled_scale: %lf\n", sebie->model.scaled_scale);
  
  if (sebie->model.selection_width == 0 || 
      sebie->model.selection_height == 0) {
    sebie_error_popup(sebie, "No region selected.");
    return;
  }
  
  int x = (double)sebie->model.selection_x / sebie->model.scaled_scale;
  int y = (double)sebie->model.selection_y / sebie->model.scaled_scale;
  int width = (double)sebie->model.selection_width / sebie->model.scaled_scale;
  int height = (double)sebie->model.selection_height / sebie->model.scaled_scale;
  int fac_width = sebie->model.selection_fac_width;
  int fac_height = sebie->model.selection_fac_height;

  sebie_debug("with rotation:\n");
  sebie_debug("x:  %d\n", x);
  sebie_debug("y:  %d\n", y);
  sebie_debug("width:  %d\n", width);
  sebie_debug("height: %d\n", height);

  if (sebie->model.scaled_rotation == SEBIE_ROTATE_090) {
    gint tmp1 = x;
    gint tmp2 = width;
    x = y;
    width = height;
    height = tmp2;   
    y = sebie->model.image_height - tmp1 - tmp2;
    fac_width = sebie->model.selection_fac_height;
    fac_height = sebie->model.selection_fac_width;
  } else if (sebie->model.scaled_rotation == SEBIE_ROTATE_180) {
    x = sebie->model.image_width  - x - width;
    y = sebie->model.image_height - y - height;
  } else if (sebie->model.scaled_rotation == SEBIE_ROTATE_270) {
    gint tmp1 = x;
    gint tmp2 = width;
    x = sebie->model.image_width - y - height;
    width = height;
    height = tmp2;
    y = tmp1;
    fac_width = sebie->model.selection_fac_height;
    fac_height = sebie->model.selection_fac_width;
  }
  sebie_debug("without rotation:\n");
  sebie_debug("x:  %d\n", x);
  sebie_debug("y:  %d\n", y); 
  sebie_debug("width:  %d\n", width);
  sebie_debug("height: %d\n", height);

  if (x < 0 || y < 0 ||
      x + width > sebie->model.image_width ||
      y + height > sebie->model.image_height) {
    sebie_error_popup(sebie, "Selected region out of range.");
    return;
  }
  
  GdkPixbuf *sub = gdk_pixbuf_new_subpixbuf(sebie->model.image_pixbuf,
					    x, y,
					    width, height);

  GdkPixbuf *corr = pixbuf_color_corrections(sub,
					     sebie->model.image_gamma,
					     sebie->model.image_grayscale);

  GdkPixbuf *save = gdk_pixbuf_scale_simple(corr, 
					    fac_width,
					    fac_height,
					    GDK_INTERP_HYPER);

  if (corr != NULL) {
    g_object_unref(corr);
  }
  
  if (sebie->model.scaled_rotation == SEBIE_ROTATE_090 ||
      sebie->model.scaled_rotation == SEBIE_ROTATE_180 ||
      sebie->model.scaled_rotation == SEBIE_ROTATE_270 ) {
    GdkPixbuf *rotated = rotate_pixbuf(save, sebie->model.scaled_rotation);
    if (save != NULL) {
      g_object_unref(save);
    }
    save = rotated;
  }
  GError *error = NULL;
  
  if (gdk_pixbuf_save(save, sebie->model.output_file_name, "jpeg", &error, "quality", "85", NULL) == FALSE) {
    sebie_error_popup(sebie, "gdb_pixbuf_save failed.");
  }

  if (save != NULL) {
    g_object_unref(save);
  }
  if (sub != NULL) {
    g_object_unref(sub);
  }
}


void
on_button_rotate_left_clicked (GtkButton *button,
			       gpointer  user_data)
{
  struct sebie_s *sebie = (struct sebie_s *)user_data;
  sebie->model.scaled_rotation = (sebie->model.scaled_rotation + SEBIE_ROTATE_270) % SEBIE_ROTATE_360;
  rotate_scaled(sebie, SEBIE_ROTATE_270);
}

void
on_button_rotate_right_clicked (GtkButton *button,
				gpointer  user_data)
{
  struct sebie_s *sebie = (struct sebie_s *)user_data;
  sebie->model.scaled_rotation = (sebie->model.scaled_rotation + SEBIE_ROTATE_090) % SEBIE_ROTATE_360;
  rotate_scaled(sebie, SEBIE_ROTATE_090);
}

void
on_entry_selection_fac_width_changed (GtkEditable *editable,
				      gpointer  user_data)
{
  struct sebie_s *sebie = (struct sebie_s *)user_data;
  if (sebie->dirty_tags == 0) {
    G_CONST_RETURN gchar *fac_width = gtk_entry_get_text(GTK_ENTRY(editable));
    sebie_debug("fac_width = %s %d\n", fac_width, atoi(fac_width));
    sebie->model.selection_fac_width = atoi(fac_width);
    sebie->model_tags |= SEBIE_MODEL_UPDATE_SELECTION_FAC;
    sebie->dirty_tags |= SEBIE_UPDATE_SELECTION_FAC;
    sebie_update_view(sebie);
  }
}

void
on_entry_selection_fac_height_changed (GtkEditable *editable,
				       gpointer  user_data)
{
  struct sebie_s *sebie = (struct sebie_s *)user_data;
  if (sebie->dirty_tags == 0) {
    G_CONST_RETURN gchar *fac_height = gtk_entry_get_text(GTK_ENTRY(editable));
    sebie_debug("fac_height = %s %d\n", fac_height, atoi(fac_height));
    sebie->model.selection_fac_height = atoi(fac_height);
    sebie->model_tags |= SEBIE_MODEL_UPDATE_SELECTION_FAC;
    sebie->dirty_tags |= SEBIE_UPDATE_SELECTION_FAC;
    sebie_update_view(sebie);
  }
}

void
on_button_fac_swap (GtkButton *button,
		    gpointer  user_data)
{
  struct sebie_s *sebie = (struct sebie_s *)user_data;
  gint tmp = sebie->model.selection_fac_width;
  sebie->model.selection_fac_width = sebie->model.selection_fac_height;
  sebie->model.selection_fac_height = tmp;
  sebie->model_tags |= SEBIE_MODEL_UPDATE_SELECTION_FAC;
  sebie->dirty_tags |= SEBIE_UPDATE_SELECTION_FAC;
  sebie->dirty_tags |= SEBIE_UPDATE_SELECTION_FAC_W;
  sebie->dirty_tags |= SEBIE_UPDATE_SELECTION_FAC_H;
  sebie_update_view(sebie);
}

void
on_entry_selection_gamma_changed (GtkEditable *editable,
				  gpointer  user_data)
{
  struct sebie_s *sebie = (struct sebie_s *)user_data;
  G_CONST_RETURN gchar *gamma_str = gtk_entry_get_text(GTK_ENTRY(editable));
  sebie_debug("image_gamma_str = '%s'\n", gamma_str);
  double new_gamma = 0.0;
  if (sscanf(gamma_str, "%lf", &new_gamma) != 1) {
    sebie_debug("no valid gamma input\n");
    return;
  }
  if (new_gamma < 0.1 || new_gamma > 3.0) {
    sebie_debug("new_gamma %lf out of range\n", new_gamma);
    return;
  }
  if (new_gamma == sebie->model.image_gamma) {
    sebie_debug("new_gamma %lf == old_gamma\n", new_gamma);
    return;
  }
  sebie_debug("set new image_gamma = %f\n", new_gamma);
  sebie->model.image_gamma = new_gamma;
  sebie->model_tags |= SEBIE_MODEL_UPDATE_GAMMA;
  sebie->dirty_tags |= SEBIE_UPDATE_IMAGE_GAMMA;
  apply_color_corrections(sebie);
}

void
on_button_grayscale_clicked(GtkToggleButton *button,
			    gpointer  user_data)
{
  struct sebie_s *sebie = (struct sebie_s *)user_data;
  sebie->model.image_grayscale = gtk_toggle_button_get_active (button);
  sebie_debug("set new image_grayscale = %d\n", sebie->model.image_grayscale);
  sebie->model_tags |= SEBIE_MODEL_UPDATE_GRAYSCALE;
  apply_color_corrections(sebie);
}

gboolean
on_drawingarea1_expose_event (GtkWidget       *widget,
			      GdkEventExpose  *event,
			      gpointer         user_data)
{
  struct sebie_s *sebie = (struct sebie_s *)user_data;
  sebie->dirty_tags |= SEBIE_UPDATE_SELECTION_RECT;
  sebie_update_view(sebie);
  return FALSE;
}

static gboolean
in_selected_area(gint x, gint y, struct sebie_s *sebie)
{
  if (x >= sebie->model.selection_x &&
      x <= sebie->model.selection_x + sebie->model.selection_width &&
      y >= sebie->model.selection_y && 
      y <= sebie->model.selection_y + sebie->model.selection_height) {
    return TRUE;
  } else {
    return FALSE;
  }
}

static gboolean
in_upper_left_corner(gint x, gint y, struct sebie_s *sebie)
{
  if (x < sebie->model.selection_x &&
      x > sebie->model.selection_x-SEBIE_CATCH_PIXEL_COUNT &&
      y < sebie->model.selection_y &&
      y > sebie->model.selection_y-SEBIE_CATCH_PIXEL_COUNT) {
    return TRUE;
  } else {
    return FALSE;
  }
}

static gboolean
in_upper_right_corner(gint x, gint y, struct sebie_s *sebie)
{
  if (x > sebie->model.selection_x+sebie->model.selection_width &&
      x < sebie->model.selection_x+sebie->model.selection_width+SEBIE_CATCH_PIXEL_COUNT &&
      y < sebie->model.selection_y &&
      y > sebie->model.selection_y-SEBIE_CATCH_PIXEL_COUNT) {
    return TRUE;
  } else {
    return FALSE;
  }
}

static gboolean
in_lower_right_corner(gint x, gint y, struct sebie_s *sebie)
{
  if (x > sebie->model.selection_x+sebie->model.selection_width &&
      x < sebie->model.selection_x+sebie->model.selection_width+SEBIE_CATCH_PIXEL_COUNT &&
      y > sebie->model.selection_y+sebie->model.selection_height &&
      y < sebie->model.selection_y+sebie->model.selection_height+SEBIE_CATCH_PIXEL_COUNT) {
    return TRUE;
  } else {
    return FALSE;
  }
}

static gboolean
in_lower_left_corner(gint x, gint y, struct sebie_s *sebie)
{
  if (x < sebie->model.selection_x &&
      x > sebie->model.selection_x-SEBIE_CATCH_PIXEL_COUNT &&
      y > sebie->model.selection_y+sebie->model.selection_height &&
      y < sebie->model.selection_y+sebie->model.selection_height+SEBIE_CATCH_PIXEL_COUNT) {
    return TRUE;
  } else {
    return FALSE;
  }
}

gboolean
on_drawingarea1_button_press_event (GtkWidget       *widget,
				    GdkEventButton  *event,
				    gpointer         user_data)
{
  struct sebie_s *sebie = (struct sebie_s *)user_data;

  if (event->button == 1) { /* left button */
    sebie_debug("button_press_event %f %f\n", event->x, event->y);
    if (in_upper_left_corner(event->x, event->y, sebie)) {
      sebie_debug("UPPER_LEFT\n");
      sebie->model.mouse_click_type = SEBIE_RESIZE_SELECTION;
      sebie->model.resize_corner = SEBIE_UPPER_LEFT_CORNER;
    } else if (in_upper_right_corner(event->x, event->y, sebie)) {
      sebie_debug("UPPER_RIGHT\n");
      sebie->model.mouse_click_type = SEBIE_RESIZE_SELECTION;
      sebie->model.resize_corner = SEBIE_UPPER_RIGHT_CORNER;
    } else if (in_lower_right_corner(event->x, event->y, sebie)) {
      sebie_debug("LOWER_RIGHT\n");
      sebie->model.mouse_click_type = SEBIE_RESIZE_SELECTION;
      sebie->model.resize_corner = SEBIE_LOWER_RIGHT_CORNER;
    } else if (in_lower_left_corner(event->x, event->y, sebie)) {
      sebie_debug("LOWER_LEFT\n");
      sebie->model.mouse_click_type = SEBIE_RESIZE_SELECTION;
      sebie->model.resize_corner = SEBIE_LOWER_LEFT_CORNER;
    } else if (in_selected_area(event->x, event->y, sebie)) {
      sebie_debug("MOVE\n");
      sebie->model.mouse_click_type = SEBIE_MOVE_SELECTION;
      sebie->model.mouse_pos_lx = event->x;
      sebie->model.mouse_pos_ly = event->y;
    } else {
      sebie_debug("LOWER_RIGHT NEW\n");
      sebie->model.mouse_click_type = SEBIE_RESIZE_SELECTION;
      sebie->model.resize_corner = SEBIE_LOWER_RIGHT_CORNER;
      sebie->model.selection_x = event->x;
      sebie->model.selection_y = event->y;
      sebie->model.selection_width  = 0;
      sebie->model.selection_height = 0;
    }
    
    sebie->dirty_tags |= SEBIE_UPDATE_SELECTION_RECT;
    sebie_update_view(sebie);
  }
  
  return FALSE;
}

gboolean
on_drawingarea1_motion_notify_event (GtkWidget       *widget,
				     GdkEventMotion  *event,
				     gpointer         user_data)
{
  struct sebie_s *sebie = (struct sebie_s *)user_data;
  
  gint x;
  gint y;
  GdkModifierType state;
  
  if (event->is_hint) {
    gdk_window_get_pointer (event->window, &x, &y, &state);
  } else {
    x = event->x;
    y = event->y;
    state = event->state;
  }

  if (state & GDK_BUTTON1_MASK) {
    sebie_debug("motion_notify_event %d %d \n", x, y);
    if (sebie->model.mouse_click_type == SEBIE_RESIZE_SELECTION) {
      sebie_debug("resize selection\n");
      gint x1 = sebie->model.selection_x;
      gint y1 = sebie->model.selection_y;
      gint x2 = sebie->model.selection_x + sebie->model.selection_width;
      gint y2 = sebie->model.selection_y + sebie->model.selection_height;

      if (sebie->model.resize_corner == SEBIE_UPPER_LEFT_CORNER) {
	gint nwidth  = sebie->model.selection_x + sebie->model.selection_width - x;
	gint nheight = sebie->model.selection_y + sebie->model.selection_height - y;
	double nfrac = (double)nwidth / (double)nheight;
	if (nfrac >= sebie->model.selection_fac) {
	  sebie->model.selection_x = x2 - (double)nheight * sebie->model.selection_fac;
	  sebie->model.selection_y = y;
	  sebie->model.selection_width  = x2 - sebie->model.selection_x;
	  sebie->model.selection_height = y2 - sebie->model.selection_y;
	  
	} else {
	  sebie->model.selection_x = x;
	  sebie->model.selection_y = y2 - (double)nwidth / sebie->model.selection_fac;
	  sebie->model.selection_width = x2 - x;
	  sebie->model.selection_height = y2 - sebie->model.selection_y;
	}
      } else if (sebie->model.resize_corner == SEBIE_UPPER_RIGHT_CORNER) {
	gint nwidth  = x - sebie->model.selection_x;
	gint nheight = sebie->model.selection_y + sebie->model.selection_height - y;
	double nfrac = (double)nwidth / (double)nheight;
	if (nfrac >= sebie->model.selection_fac) {
	  sebie->model.selection_width = (double)nheight * sebie->model.selection_fac;
	  sebie->model.selection_y = y;
	  sebie->model.selection_height = y2 - sebie->model.selection_y;
	} else {
	  sebie->model.selection_width = nwidth;
	  sebie->model.selection_height = (double)nwidth / sebie->model.selection_fac;
	  sebie->model.selection_y = y2 - sebie->model.selection_height;
	}
      } else if (sebie->model.resize_corner == SEBIE_LOWER_RIGHT_CORNER) {
	gint nwidth  = x - sebie->model.selection_x;
	gint nheight = y - sebie->model.selection_y;
	double nfrac = (double)nwidth / (double)nheight;
	if (nfrac >= sebie->model.selection_fac) {
	  sebie->model.selection_width = (double)nheight * sebie->model.selection_fac;
	  sebie->model.selection_height = nheight;
	} else {
	  sebie->model.selection_width = nwidth;
	  sebie->model.selection_height = (double)nwidth / sebie->model.selection_fac; 
	}
      } else { /* SEBIE_LOWER_LEFT_CORNER */
	gint nwidth  = sebie->model.selection_x + sebie->model.selection_width - x;
	gint nheight = y - sebie->model.selection_y;
	double nfrac = (double)nwidth / (double)nheight;
	if (nfrac >= sebie->model.selection_fac) {
	  sebie->model.selection_width = (double)nheight * sebie->model.selection_fac;
	  sebie->model.selection_height = nheight;
	  sebie->model.selection_x = x2 - sebie->model.selection_width;
	} else {
	  sebie->model.selection_x = x;
	  sebie->model.selection_height = (double)nwidth / sebie->model.selection_fac;
	  sebie->model.selection_width = x2 - sebie->model.selection_x;
	}
      }
      /* adjust if selection crossed the fix point */
      if (sebie->model.selection_width < 0 &&
	  sebie->model.selection_height < 0) {
	sebie->model.selection_x += sebie->model.selection_width;
	sebie->model.selection_width = sebie->model.selection_width * (-1);
	sebie->model.selection_y += sebie->model.selection_height;
	sebie->model.selection_height = sebie->model.selection_height * (-1);
	switch(sebie->model.resize_corner) {
	  case SEBIE_UPPER_LEFT_CORNER: sebie->model.resize_corner = SEBIE_LOWER_RIGHT_CORNER; break;
	  case SEBIE_UPPER_RIGHT_CORNER: sebie->model.resize_corner = SEBIE_LOWER_LEFT_CORNER; break;
	  case SEBIE_LOWER_RIGHT_CORNER: sebie->model.resize_corner = SEBIE_UPPER_LEFT_CORNER; break;
	  case SEBIE_LOWER_LEFT_CORNER: sebie->model.resize_corner = SEBIE_UPPER_RIGHT_CORNER; break;
	}
      }
    } else {
      sebie_debug("move selection\n");
      gint diff_x = x - sebie->model.mouse_pos_lx;
      gint diff_y = y - sebie->model.mouse_pos_ly;
      sebie->model.selection_x += diff_x;
      sebie->model.selection_y += diff_y;
      sebie->model.mouse_pos_lx = x;
      sebie->model.mouse_pos_ly = y;
    }
    sebie->dirty_tags |= SEBIE_UPDATE_SELECTION_RECT;
    sebie_update_view(sebie);
  }

  if (in_selected_area(x,y, sebie)) {
    GdkCursor *cursor = gdk_cursor_new(GDK_FLEUR);
    gdk_window_set_cursor(event->window, cursor);
  } else if (in_upper_left_corner(x, y, sebie) ||
	     ((state & GDK_BUTTON1_MASK) &&
	      sebie->model.resize_corner == SEBIE_UPPER_LEFT_CORNER)) {
    GdkCursor *cursor = gdk_cursor_new(GDK_TOP_LEFT_CORNER);
    gdk_window_set_cursor(event->window, cursor);
  } else if (in_upper_right_corner(x, y, sebie) ||
	     ((state & GDK_BUTTON1_MASK) &&
	      sebie->model.resize_corner == SEBIE_UPPER_RIGHT_CORNER)) {
    GdkCursor *cursor = gdk_cursor_new(GDK_TOP_RIGHT_CORNER);
    gdk_window_set_cursor(event->window, cursor);
  } else if (in_lower_right_corner(x, y, sebie) ||
	     ((state & GDK_BUTTON1_MASK) &&
	      sebie->model.resize_corner == SEBIE_LOWER_RIGHT_CORNER)) {
    GdkCursor *cursor = gdk_cursor_new(GDK_BOTTOM_RIGHT_CORNER);
    gdk_window_set_cursor(event->window, cursor);
  } else if (in_lower_left_corner(x, y, sebie) ||
	     ((state & GDK_BUTTON1_MASK) &&
	      sebie->model.resize_corner == SEBIE_LOWER_LEFT_CORNER)) {
    GdkCursor *cursor = gdk_cursor_new(GDK_BOTTOM_LEFT_CORNER);
    gdk_window_set_cursor(event->window, cursor);
  } else {
    gdk_window_set_cursor (event->window, NULL);
  }

  return TRUE;
}
