/**
 *  Copyright 2002-2022 Peter Seiderer <Peter.Seiderer@ciselant.de>
 *
 *  This file is part of SeBIE.
 *
 *  SeBIE is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  SeBIE is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with SeBIE; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
#include "sebie_callbacks.h"
#include "sebie_helpers.h"
#include "sebie_gtk_helpers.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#define SEBIE_CATCH_PIXEL_COUNT  15


void
on_treeview_file_list_changed(GtkTreeSelection *selection,
                              gpointer user_data)
{
    struct sebie_s *sebie = (struct sebie_s *)user_data;
    GtkTreeIter iter;
    GtkTreeModel *model;
    gchar *filename;
    if (gtk_tree_selection_get_selected(selection, &model, &iter)) {
        gtk_tree_model_get(model, &iter, 0, &filename, -1);
        sebie_set_string(&sebie->model.input_file_name, filename);
        g_free(filename);
        load_pixbuf(sebie);
        if (sebie->model.regex_status) {
            int nmatch = 2;
            regmatch_t matchptr[nmatch];
            if (regexec(&sebie->model.regex, sebie->model.input_file_name, nmatch, matchptr, 0) == 0) {
                char buf_regex[nmatch][256];
                int i;
                for (i = 0; i < nmatch; i++) {
                    strncpy(buf_regex[i],sebie->model.input_file_name +  matchptr[i].rm_so, matchptr[i].rm_eo - matchptr[i].rm_so);
                    buf_regex[i][matchptr[i].rm_eo - matchptr[i].rm_so] = '\0';
                    sebie_debug("buf_regex[%d] = %s\n", i, buf_regex[i]);
                }
              char buf[256];
              snprintf(buf, 256, sebie->model.output_format, buf_regex[1]);
              sebie_set_string(&sebie->model.output_file_name, buf);
              sebie->dirty_tags |= SEBIE_UPDATE_OUTPUT_FILE_NAME;
#if GTK_CHECK_VERSION(4,0,0)
              sebie_update_view(sebie, NULL, NULL, 0, 0);
#else
              sebie_update_view(sebie);
#endif
          }
      } else {
          sebie_debug("sebie->model.regex_status invalid");
      }
  }
}

void
on_entry_input_path_changed (GtkEditable *editable,
                 gpointer    user_data)
{
  struct sebie_s *sebie = (struct sebie_s *)user_data;
#if GTK_CHECK_VERSION(4,0,0)
  GtkEntryBuffer* entry_buffer = gtk_entry_get_buffer(GTK_ENTRY(editable));
  const gchar *input_path = gtk_entry_buffer_get_text(entry_buffer);
#else
  const gchar *input_path = gtk_entry_get_text(GTK_ENTRY(editable));
#endif
  sebie_set_string(&sebie->model.input_path, input_path);
  sebie->model_tags |= SEBIE_MODEL_INPUT_PATH;
  sebie_update_file_list(sebie);
}


void
on_entry_input_regex_changed (GtkEditable *editable,
                 gpointer    user_data)
{
  struct sebie_s *sebie = (struct sebie_s *)user_data;
#if GTK_CHECK_VERSION(4,0,0)
  GtkEntryBuffer* entry_buffer = gtk_entry_get_buffer(GTK_ENTRY(editable));
  const gchar *input_regex = gtk_entry_buffer_get_text(entry_buffer);
#else
  const gchar *input_regex = gtk_entry_get_text(GTK_ENTRY(editable));
#endif
  sebie_debug("input_regex = %s\n", input_regex);
  sebie_set_string(&sebie->model.input_regex, input_regex);
  sebie->model_tags |= SEBIE_MODEL_INPUT_REGEX;
  sebie_regcomp(sebie);
  sebie_update_file_list(sebie);
}

void
on_entry_output_format_changed (GtkEditable *editable,
                gpointer    user_data)
{
  struct sebie_s *sebie = (struct sebie_s *)user_data;
#if GTK_CHECK_VERSION(4,0,0)
  GtkEntryBuffer* entry_buffer = gtk_entry_get_buffer(GTK_ENTRY(editable));
  const gchar *output_format = gtk_entry_buffer_get_text(entry_buffer);
#else
  const gchar *output_format = gtk_entry_get_text(GTK_ENTRY(editable));
#endif
  sebie_debug("output_format = %s\n", output_format);
  sebie_set_string(&sebie->model.output_format, output_format);
  sebie->model_tags |= SEBIE_MODEL_OUTPUT_FORMAT;
  save_sebie_config(sebie);
}


gint
window1_delete_event (GtkWidget *widget,
              GdkEvent  *event,
              gpointer   data)
{
    sebie_debug("window1_delete_event\n");
    return FALSE;
}


void
window1_destroy (GtkWidget *widget,
         gpointer   data)
{
    sebie_debug("window1_destroy\n");

#if !GTK_CHECK_VERSION(4,0,0)
    gtk_main_quit ();
#endif
}


#if GTK_CHECK_VERSION(4,0,0)
void
drawingarea1_resize (GtkWidget *widget,
                     gint width,
                     gint height,
                     gpointer user_data)
{
    struct sebie_s *sebie = (struct sebie_s *)user_data;
    gint window1_width;
    gint window1_height;

    sebie_debug("drawingarea1_resize width = %d height = %d\n", width, height);

    gtk_window_get_default_size(GTK_WINDOW(sebie->view.window1), &window1_width, &window1_height);
    sebie->model.window1_width = window1_width;
    sebie->model.window1_height = window1_height;
    sebie->model.pixbuf_width = width;
    sebie->model.pixbuf_height = height;

    sebie->model_tags |= SEBIE_MODEL_CHANGE_PIXBUF_SIZE;
    save_sebie_config(sebie);
}
#else
void
drawingarea1_size_allocate (GtkWidget *widget,
                GtkAllocation *allocation,
                gpointer   user_data)
{
    struct sebie_s *sebie = (struct sebie_s *)user_data;
    gint window1_width;
    gint window1_height;

    sebie_debug("drawingarea1_size_allocate x = %d y = %d width = %d height = %d\n",
        allocation->x,
        allocation->y,
        allocation->width,
        allocation->height);

    gtk_window_get_size(GTK_WINDOW(sebie->view.window1), &window1_width, &window1_height);
    sebie->model.window1_width = window1_width;
    sebie->model.window1_height = window1_height;
    sebie->model.pixbuf_width = allocation->width;
    sebie->model.pixbuf_height = allocation->height;

    sebie->model_tags |= SEBIE_MODEL_CHANGE_PIXBUF_SIZE;
    save_sebie_config(sebie);
}
#endif


void
on_button_quit_clicked (GtkButton *button,
            gpointer  user_data)
{
#if GTK_CHECK_VERSION(4,0,0)
  struct sebie_s *sebie = (struct sebie_s *)user_data;

  sebie_debug("on_button_quit_clicked\n");

  gtk_window_destroy(GTK_WINDOW(sebie->view.window1));
#else
  sebie_debug("on_button_quit_clicked\n");
  gtk_main_quit ();
#endif
}


void
on_button_save_clicked (GtkButton *button,
            gpointer  user_data)
{
  struct sebie_s *sebie = (struct sebie_s *)user_data;

  sebie_debug("image_width:  %d\n", sebie->model.image_width);
  sebie_debug("image_height: %d\n", sebie->model.image_height);
  sebie_debug("image_gamma:  %lf\n", sebie->model.image_gamma);
  sebie_debug("selection_x:  %d\n", sebie->model.selection_x);
  sebie_debug("selection_y:  %d\n", sebie->model.selection_y);
  sebie_debug("selection_width:  %d\n", sebie->model.selection_width);
  sebie_debug("selection_height: %d\n", sebie->model.selection_height);
  sebie_debug("scaled_scale: %lf\n", sebie->model.scaled_scale);

  if (sebie->model.selection_width == 0 ||
      sebie->model.selection_height == 0) {
    sebie_error_popup(sebie, "No region selected.");
    return;
  }

  int x = (double)sebie->model.selection_x / sebie->model.scaled_scale;
  int y = (double)sebie->model.selection_y / sebie->model.scaled_scale;
  int width = (double)sebie->model.selection_width / sebie->model.scaled_scale;
  int height = (double)sebie->model.selection_height / sebie->model.scaled_scale;
  int fac_width = sebie->model.selection_fac_width;
  int fac_height = sebie->model.selection_fac_height;

  sebie_debug("with rotation:\n");
  sebie_debug("x:  %d\n", x);
  sebie_debug("y:  %d\n", y);
  sebie_debug("width:  %d\n", width);
  sebie_debug("height: %d\n", height);

  if (sebie->model.scaled_rotation == SEBIE_ROTATE_090) {
    gint tmp1 = x;
    gint tmp2 = width;
    x = y;
    width = height;
    height = tmp2;
    y = sebie->model.image_height - tmp1 - tmp2;
    fac_width = sebie->model.selection_fac_height;
    fac_height = sebie->model.selection_fac_width;
  } else if (sebie->model.scaled_rotation == SEBIE_ROTATE_180) {
    x = sebie->model.image_width  - x - width;
    y = sebie->model.image_height - y - height;
  } else if (sebie->model.scaled_rotation == SEBIE_ROTATE_270) {
    gint tmp1 = x;
    gint tmp2 = width;
    x = sebie->model.image_width - y - height;
    width = height;
    height = tmp2;
    y = tmp1;
    fac_width = sebie->model.selection_fac_height;
    fac_height = sebie->model.selection_fac_width;
  }
  sebie_debug("without rotation:\n");
  sebie_debug("x:  %d\n", x);
  sebie_debug("y:  %d\n", y);
  sebie_debug("width:  %d\n", width);
  sebie_debug("height: %d\n", height);

  if (x < 0 || y < 0 ||
      x + width > sebie->model.image_width ||
      y + height > sebie->model.image_height) {
    sebie_error_popup(sebie, "Selected region out of range.");
    return;
  }

  GdkPixbuf *sub = gdk_pixbuf_new_subpixbuf(sebie->model.image_pixbuf,
                        x, y,
                        width, height);

  GdkPixbuf *corr = pixbuf_color_corrections(sub,
                         sebie->model.image_gamma,
                         sebie->model.image_grayscale);

  GdkPixbuf *save = gdk_pixbuf_scale_simple(corr,
                        fac_width,
                        fac_height,
                        GDK_INTERP_HYPER);

  if (corr != NULL) {
    g_object_unref(corr);
  }

  if (sebie->model.scaled_rotation == SEBIE_ROTATE_090 ||
      sebie->model.scaled_rotation == SEBIE_ROTATE_180 ||
      sebie->model.scaled_rotation == SEBIE_ROTATE_270 ) {
    GdkPixbuf *rotated = rotate_pixbuf(save, sebie->model.scaled_rotation);
    if (save != NULL) {
      g_object_unref(save);
    }
    save = rotated;
  }
  GError *error = NULL;

  if (gdk_pixbuf_save(save, sebie->model.output_file_name, "jpeg", &error, "quality", "85", NULL) == FALSE) {
    sebie_error_popup(sebie, "gdb_pixbuf_save failed.");
  }

  if (save != NULL) {
    g_object_unref(save);
  }
  if (sub != NULL) {
    g_object_unref(sub);
  }
}


void
on_button_rotate_left_clicked (GtkButton *button,
                   gpointer  user_data)
{
  struct sebie_s *sebie = (struct sebie_s *)user_data;

  // no image loaded yet, simply do nothing
  if (sebie->model.scaled_pixbuf == NULL) {
    return;
  }

  sebie->model.scaled_rotation = (sebie->model.scaled_rotation + SEBIE_ROTATE_270) % SEBIE_ROTATE_360;
  rotate_scaled(sebie, SEBIE_ROTATE_270);
}

void
on_button_rotate_right_clicked (GtkButton *button,
                gpointer  user_data)
{
  struct sebie_s *sebie = (struct sebie_s *)user_data;

  // no image loaded yet, simply do nothing
  if (sebie->model.scaled_pixbuf == NULL) {
    return;
  }

  sebie->model.scaled_rotation = (sebie->model.scaled_rotation + SEBIE_ROTATE_090) % SEBIE_ROTATE_360;
  rotate_scaled(sebie, SEBIE_ROTATE_090);
}

void
on_entry_selection_fac_width_changed (GtkEditable *editable,
                      gpointer  user_data)
{
  struct sebie_s *sebie = (struct sebie_s *)user_data;
  if (sebie->dirty_tags == 0) {
#if GTK_CHECK_VERSION(4,0,0)
    GtkEntryBuffer* entry_buffer = gtk_entry_get_buffer(GTK_ENTRY(editable));
    const gchar *fac_width_str = gtk_entry_buffer_get_text(entry_buffer);
#else
    const gchar *fac_width_str = gtk_entry_get_text(GTK_ENTRY(editable));
#endif
    gchar *end_ptr;
    errno = 0;
    gint fac_width = strtol(fac_width_str, &end_ptr, 10);
    if (errno != 0 || end_ptr == fac_width_str || *end_ptr != '\0' || fac_width < 1) {
        sebie_debug("fac_width = %s/%d invalid\n", fac_width_str, fac_width);
        sebie->model.selection_fac_width = 1;
        sebie->model_tags |= SEBIE_MODEL_UPDATE_SELECTION_FAC;
        sebie->dirty_tags |= SEBIE_UPDATE_SELECTION_FAC_W;
    } else {
        sebie_debug("fac_width = %s %d\n", fac_width_str, fac_width);
        sebie->model.selection_fac_width = fac_width;
        sebie->model_tags |= SEBIE_MODEL_UPDATE_SELECTION_FAC;
        sebie->dirty_tags |= SEBIE_UPDATE_SELECTION_FAC;
    }
#if GTK_CHECK_VERSION(4,0,0)
    sebie_update_view(sebie, NULL, NULL, 0, 0);
#else
    sebie_update_view(sebie);
#endif
  }
}

void
on_entry_selection_fac_height_changed (GtkEditable *editable,
                       gpointer  user_data)
{
  struct sebie_s *sebie = (struct sebie_s *)user_data;
  if (sebie->dirty_tags == 0) {
#if GTK_CHECK_VERSION(4,0,0)
    GtkEntryBuffer* entry_buffer = gtk_entry_get_buffer(GTK_ENTRY(editable));
    const gchar *fac_height_str = gtk_entry_buffer_get_text(entry_buffer);
#else
    const gchar *fac_height_str = gtk_entry_get_text(GTK_ENTRY(editable));
#endif
    gchar *end_ptr;
    errno = 0;
    gint fac_height = strtol(fac_height_str, &end_ptr, 10);
    if (errno != 0 || end_ptr == fac_height_str || *end_ptr != '\0' || fac_height < 1) {
        sebie_debug("fac_height = %s/%d invalid\n", fac_height_str, fac_height);
        sebie->model.selection_fac_height = 1;
        sebie->model_tags |= SEBIE_MODEL_UPDATE_SELECTION_FAC;
        sebie->dirty_tags |= SEBIE_UPDATE_SELECTION_FAC_H;
    } else {
        sebie_debug("fac_height = %s/%d\n", fac_height_str, fac_height);
        sebie->model.selection_fac_height = fac_height;
        sebie->model_tags |= SEBIE_MODEL_UPDATE_SELECTION_FAC;
        sebie->dirty_tags |= SEBIE_UPDATE_SELECTION_FAC;
    }
#if GTK_CHECK_VERSION(4,0,0)
    sebie_update_view(sebie, NULL, NULL, 0, 0);
#else
    sebie_update_view(sebie);
#endif
  }
}

void
on_button_fac_swap (GtkButton *button,
            gpointer  user_data)
{
  struct sebie_s *sebie = (struct sebie_s *)user_data;
  gint tmp = sebie->model.selection_fac_width;
  sebie->model.selection_fac_width = sebie->model.selection_fac_height;
  sebie->model.selection_fac_height = tmp;
  sebie->model_tags |= SEBIE_MODEL_UPDATE_SELECTION_FAC;
  sebie->dirty_tags |= SEBIE_UPDATE_SELECTION_FAC;
  sebie->dirty_tags |= SEBIE_UPDATE_SELECTION_FAC_W;
  sebie->dirty_tags |= SEBIE_UPDATE_SELECTION_FAC_H;
#if GTK_CHECK_VERSION(4,0,0)
  sebie_update_view(sebie, NULL, NULL, 0, 0);
#else
  sebie_update_view(sebie);
#endif
}

void
on_entry_selection_gamma_changed (GtkEditable *editable,
                  gpointer  user_data)
{
  struct sebie_s *sebie = (struct sebie_s *)user_data;
#if GTK_CHECK_VERSION(4,0,0)
  double new_gamma = gtk_spin_button_get_value(GTK_SPIN_BUTTON(editable));
#else
  const gchar *gamma_str = gtk_entry_get_text(GTK_ENTRY(editable));
  sebie_debug("image_gamma_str = '%s'\n", gamma_str);
  double new_gamma = 0.0;
  if (sscanf(gamma_str, "%lf", &new_gamma) != 1) {
    sebie_debug("no valid gamma input\n");
    return;
  }
#endif
  if (new_gamma < 0.1 || new_gamma > 3.0) {
    sebie_debug("new_gamma %lf out of range\n", new_gamma);
    return;
  }
  if (new_gamma == sebie->model.image_gamma) {
    sebie_debug("new_gamma %lf == old_gamma\n", new_gamma);
    return;
  }
  sebie_debug("set new image_gamma = %f\n", new_gamma);
  sebie->model.image_gamma = new_gamma;
  sebie->model_tags |= SEBIE_MODEL_UPDATE_GAMMA;
  apply_color_corrections(sebie);
}


#if GTK_CHECK_VERSION(4,0,0)
void
on_button_grayscale_clicked(GtkToggleButton *button,
                            gpointer user_data)
{
  struct sebie_s *sebie = (struct sebie_s *)user_data;
  sebie->model.image_grayscale = gtk_toggle_button_get_active (button);
  sebie_debug("set new image_grayscale = %d\n", sebie->model.image_grayscale);
  sebie->model_tags |= SEBIE_MODEL_UPDATE_GRAYSCALE;
  apply_color_corrections(sebie);
}
#elif GTK_CHECK_VERSION(3,0,0)
void
on_button_grayscale_clicked(GtkToolItem *button,
                gpointer  user_data)
{
  struct sebie_s *sebie = (struct sebie_s *)user_data;
  sebie->model.image_grayscale = gtk_toggle_tool_button_get_active (GTK_TOGGLE_TOOL_BUTTON(button));
  sebie_debug("set new image_grayscale = %d\n", sebie->model.image_grayscale);
  sebie->model_tags |= SEBIE_MODEL_UPDATE_GRAYSCALE;
  apply_color_corrections(sebie);
}
#else
void
on_button_grayscale_clicked(GtkToggleButton *button,
                gpointer  user_data)
{
  struct sebie_s *sebie = (struct sebie_s *)user_data;
  sebie->model.image_grayscale = gtk_toggle_button_get_active (button);
  sebie_debug("set new image_grayscale = %d\n", sebie->model.image_grayscale);
  sebie->model_tags |= SEBIE_MODEL_UPDATE_GRAYSCALE;
  apply_color_corrections(sebie);
}
#endif


#if GTK_CHECK_VERSION(4,0,0)
void
drawingarea1_draw_function(GtkDrawingArea *area,
                           cairo_t *cr,
                           int width,
                           int height,
                           gpointer user_data)
{
  struct sebie_s *sebie = (struct sebie_s *)user_data;
  sebie->dirty_tags |= SEBIE_UPDATE_SCALED_PIXBUF;
  sebie->dirty_tags |= SEBIE_UPDATE_SELECTION_RECT;
  sebie_update_view(sebie, area, cr, width, height);
}
#else
gboolean
on_drawingarea1_expose_event (GtkWidget       *widget,
                  GdkEventExpose  *event,
                  gpointer         user_data)
{
  struct sebie_s *sebie = (struct sebie_s *)user_data;
#if GTK_CHECK_VERSION(3,0,0)
  sebie->dirty_tags |= SEBIE_UPDATE_SCALED_PIXBUF;
#endif
  sebie->dirty_tags |= SEBIE_UPDATE_SELECTION_RECT;
  sebie_update_view(sebie);
  return FALSE;
}
#endif


static gboolean
in_selected_area(gint x, gint y, struct sebie_s *sebie)
{
  sebie_debug("in_selected_area: selection_y: %d selection_y: %d selection_width: %d selection_height: %d x: %d y: %d",
              sebie->model.selection_x, sebie->model.selection_y, sebie->model.selection_width, sebie->model.selection_height, x, y);

  if (x >= sebie->model.selection_x &&
      x <= sebie->model.selection_x + sebie->model.selection_width &&
      y >= sebie->model.selection_y &&
      y <= sebie->model.selection_y + sebie->model.selection_height) {
    sebie_debug(" TRUE\n");
    return TRUE;
  } else {
    sebie_debug(" FALSE\n");
    return FALSE;
  }
}

static gboolean
in_upper_left_corner(gint x, gint y, struct sebie_s *sebie)
{
  sebie_debug("in_upper_left_corner: selection_y: %d selection_y: %d selection_width: %d selection_height: %d x: %d y: %d",
              sebie->model.selection_x, sebie->model.selection_y, sebie->model.selection_width, sebie->model.selection_height, x, y);

  if (x < sebie->model.selection_x &&
      x > sebie->model.selection_x-SEBIE_CATCH_PIXEL_COUNT &&
      y < sebie->model.selection_y &&
      y > sebie->model.selection_y-SEBIE_CATCH_PIXEL_COUNT) {
    sebie_debug(" TRUE\n");
    return TRUE;
  } else {
    sebie_debug(" FALSE\n");
    return FALSE;
  }
}

static gboolean
in_upper_right_corner(gint x, gint y, struct sebie_s *sebie)
{
  sebie_debug("in_upper_right_corner: selection_y: %d selection_y: %d selection_width: %d selection_height: %d x: %d y: %d",
              sebie->model.selection_x, sebie->model.selection_y, sebie->model.selection_width, sebie->model.selection_height, x, y);

  if (x > sebie->model.selection_x+sebie->model.selection_width &&
      x < sebie->model.selection_x+sebie->model.selection_width+SEBIE_CATCH_PIXEL_COUNT &&
      y < sebie->model.selection_y &&
      y > sebie->model.selection_y-SEBIE_CATCH_PIXEL_COUNT) {
    sebie_debug(" TRUE\n");
    return TRUE;
  } else {
    sebie_debug(" FALSE\n");
    return FALSE;
  }
}

static gboolean
in_lower_right_corner(gint x, gint y, struct sebie_s *sebie)
{
  sebie_debug("in_lower_right_corner: selection_y: %d selection_y: %d selection_width: %d selection_height: %d x: %d y: %d",
              sebie->model.selection_x, sebie->model.selection_y, sebie->model.selection_width, sebie->model.selection_height, x, y);

  if (x > sebie->model.selection_x+sebie->model.selection_width &&
      x < sebie->model.selection_x+sebie->model.selection_width+SEBIE_CATCH_PIXEL_COUNT &&
      y > sebie->model.selection_y+sebie->model.selection_height &&
      y < sebie->model.selection_y+sebie->model.selection_height+SEBIE_CATCH_PIXEL_COUNT) {
    sebie_debug(" TRUE\n");
    return TRUE;
  } else {
    sebie_debug(" FALSE\n");
    return FALSE;
  }
}

static gboolean
in_lower_left_corner(gint x, gint y, struct sebie_s *sebie)
{
  sebie_debug("in_lower_left_corner: selection_y: %d selection_y: %d selection_width: %d selection_height: %d x: %d y: %d",
              sebie->model.selection_x, sebie->model.selection_y, sebie->model.selection_width, sebie->model.selection_height, x, y);

  if (x < sebie->model.selection_x &&
      x > sebie->model.selection_x-SEBIE_CATCH_PIXEL_COUNT &&
      y > sebie->model.selection_y+sebie->model.selection_height &&
      y < sebie->model.selection_y+sebie->model.selection_height+SEBIE_CATCH_PIXEL_COUNT) {
    sebie_debug(" TRUE\n");
    return TRUE;
  } else {
    sebie_debug(" FALSE\n");
    return FALSE;
  }
}

#if GTK_CHECK_VERSION(4,0,0)
void
on_drawingarea1_drag_begin(GtkGestureDrag *gesture,
                           double x,
                           double y,
                           gpointer user_data)
{
    struct sebie_s *sebie = (struct sebie_s *)user_data;

    sebie_debug("on_drawingarea1_drag_begin: x: %f y: %f\n", x, y);

    if (in_upper_left_corner(x, y, sebie)) {
        sebie_debug("UPPER_LEFT\n");
        sebie->model.mouse_click_type = SEBIE_RESIZE_SELECTION;
        sebie->model.resize_corner = SEBIE_UPPER_LEFT_CORNER;
    } else if (in_upper_right_corner(x, y, sebie)) {
        sebie_debug("UPPER_RIGHT\n");
        sebie->model.mouse_click_type = SEBIE_RESIZE_SELECTION;
        sebie->model.resize_corner = SEBIE_UPPER_RIGHT_CORNER;
    } else if (in_lower_right_corner(x, y, sebie)) {
        sebie_debug("LOWER_RIGHT\n");
        sebie->model.mouse_click_type = SEBIE_RESIZE_SELECTION;
        sebie->model.resize_corner = SEBIE_LOWER_RIGHT_CORNER;
    } else if (in_lower_left_corner(x, y, sebie)) {
        sebie_debug("LOWER_LEFT\n");
        sebie->model.mouse_click_type = SEBIE_RESIZE_SELECTION;
        sebie->model.resize_corner = SEBIE_LOWER_LEFT_CORNER;
    } else if (in_selected_area(x, y, sebie)) {
        sebie_debug("MOVE\n");
        sebie->model.mouse_click_type = SEBIE_MOVE_SELECTION;
        sebie->model.mouse_pos_lx = x - sebie->model.selection_x;
        sebie->model.mouse_pos_ly = y - sebie->model.selection_y;
    } else {
        sebie_debug("LOWER_RIGHT NEW\n");
        sebie->model.mouse_click_type = SEBIE_RESIZE_SELECTION;
        sebie->model.resize_corner = SEBIE_LOWER_RIGHT_CORNER;
        sebie->model.selection_x = x;
        sebie->model.selection_y = y;
        sebie->model.selection_width  = 0;
        sebie->model.selection_height = 0;
    }

    sebie->dirty_tags |= SEBIE_UPDATE_SELECTION_RECT;
    sebie_update_view(sebie, NULL, NULL, 0, 0);
}


static void
drawingarea1_update_cursor(gint x, gint y, struct sebie_s* sebie)
{
    if (in_selected_area(x,y, sebie) || sebie->model.mouse_click_type == SEBIE_MOVE_SELECTION) {
        sebie_debug("MOVE\n");
        gtk_widget_set_cursor_from_name(sebie->view.drawingarea1, "move");
    } else if (in_upper_left_corner(x, y, sebie) ||
               (sebie->model.mouse_click_type == SEBIE_RESIZE_SELECTION && sebie->model.resize_corner == SEBIE_UPPER_LEFT_CORNER)) {
        sebie_debug("UPPER_LEFT\n");
        gtk_widget_set_cursor_from_name(sebie->view.drawingarea1, "nw-resize");
    } else if (in_upper_right_corner(x, y, sebie) ||
               (sebie->model.mouse_click_type == SEBIE_RESIZE_SELECTION && sebie->model.resize_corner == SEBIE_UPPER_RIGHT_CORNER)) {
        sebie_debug("UPPER_RIGHT\n");
        gtk_widget_set_cursor_from_name(sebie->view.drawingarea1, "ne-resize");
    } else if (in_lower_right_corner(x, y, sebie) ||
               (sebie->model.mouse_click_type == SEBIE_RESIZE_SELECTION && sebie->model.resize_corner == SEBIE_LOWER_RIGHT_CORNER)) {
        sebie_debug("LOWER_RIGHT\n");
        gtk_widget_set_cursor_from_name(sebie->view.drawingarea1, "se-resize");
    } else if (in_lower_left_corner(x, y, sebie) ||
               (sebie->model.mouse_click_type == SEBIE_RESIZE_SELECTION && sebie->model.resize_corner == SEBIE_LOWER_LEFT_CORNER)) {
        sebie_debug("LOWER_LEFT\n");
        gtk_widget_set_cursor_from_name(sebie->view.drawingarea1, "sw-resize");
    } else {
        sebie_debug("DEFAULT\n");
        gtk_widget_set_cursor_from_name(sebie->view.drawingarea1, "default");
    }
}


void
on_drawingarea1_drag_update(GtkGestureDrag *gesture,
                            double xmove,
                            double ymove,
                            gpointer user_data)
{
    struct sebie_s *sebie = (struct sebie_s *)user_data;

    sebie_debug("on_drawingarea1_drag_update: xmove: %f ymove: %f\n", xmove, ymove);

    gint x = sebie->model.selection_x + xmove + 0.5;
    gint y = sebie->model.selection_y + ymove + 0.5;

    if (sebie->model.mouse_click_type == SEBIE_RESIZE_SELECTION) {
        sebie_debug("resize selection\n");
#if 0
        gint x1 = sebie->model.selection_x;
        gint y1 = sebie->model.selection_y;
#endif
        gint x2 = sebie->model.selection_x + sebie->model.selection_width;
        gint y2 = sebie->model.selection_y + sebie->model.selection_height;

        if (sebie->model.resize_corner == SEBIE_UPPER_LEFT_CORNER) {
            gint nwidth  = sebie->model.selection_x + sebie->model.selection_width - x;
            gint nheight = sebie->model.selection_y + sebie->model.selection_height - y;
            double nfrac = (double)nwidth / (double)nheight;
            if (nfrac >= sebie->model.selection_fac) {
                sebie->model.selection_x = x2 - (double)nheight * sebie->model.selection_fac;
                sebie->model.selection_y = y;
                sebie->model.selection_width  = x2 - sebie->model.selection_x;
                sebie->model.selection_height = y2 - sebie->model.selection_y;
            } else {
                sebie->model.selection_x = x;
                sebie->model.selection_y = y2 - (double)nwidth / sebie->model.selection_fac;
                sebie->model.selection_width = x2 - x;
                sebie->model.selection_height = y2 - sebie->model.selection_y;
            }
        } else if (sebie->model.resize_corner == SEBIE_UPPER_RIGHT_CORNER) {
            gint nwidth  = x - sebie->model.selection_x;
            gint nheight = sebie->model.selection_y + sebie->model.selection_height - y;
            double nfrac = (double)nwidth / (double)nheight;
            if (nfrac >= sebie->model.selection_fac) {
                sebie->model.selection_width = (double)nheight * sebie->model.selection_fac;
                sebie->model.selection_y = y;
                sebie->model.selection_height = y2 - sebie->model.selection_y;
            } else {
                sebie->model.selection_width = nwidth;
                sebie->model.selection_height = (double)nwidth / sebie->model.selection_fac;
                sebie->model.selection_y = y2 - sebie->model.selection_height;
            }
        } else if (sebie->model.resize_corner == SEBIE_LOWER_RIGHT_CORNER) {
            gint nwidth  = x - sebie->model.selection_x;
            gint nheight = y - sebie->model.selection_y;
            double nfrac = (double)nwidth / (double)nheight;
            if (nfrac >= sebie->model.selection_fac) {
                sebie->model.selection_width = (double)nheight * sebie->model.selection_fac;
                sebie->model.selection_height = nheight;
            } else {
                sebie->model.selection_width = nwidth;
                sebie->model.selection_height = (double)nwidth / sebie->model.selection_fac;
            }
        } else { /* SEBIE_LOWER_LEFT_CORNER */
            gint nwidth  = sebie->model.selection_x + sebie->model.selection_width - x;
            gint nheight = y - sebie->model.selection_y;
            double nfrac = (double)nwidth / (double)nheight;
            if (nfrac >= sebie->model.selection_fac) {
                sebie->model.selection_width = (double)nheight * sebie->model.selection_fac;
                sebie->model.selection_height = nheight;
                sebie->model.selection_x = x2 - sebie->model.selection_width;
            } else {
                sebie->model.selection_x = x;
                sebie->model.selection_height = (double)nwidth / sebie->model.selection_fac;
                sebie->model.selection_width = x2 - sebie->model.selection_x;
            }
        }
        /* adjust if selection crossed the fix point */
        if (sebie->model.selection_width < 0 &&
            sebie->model.selection_height < 0) {
            sebie->model.selection_x += sebie->model.selection_width;
            sebie->model.selection_width = sebie->model.selection_width * (-1);
            sebie->model.selection_y += sebie->model.selection_height;
            sebie->model.selection_height = sebie->model.selection_height * (-1);
            switch(sebie->model.resize_corner) {
                case SEBIE_UPPER_LEFT_CORNER: sebie->model.resize_corner = SEBIE_LOWER_RIGHT_CORNER; break;
                case SEBIE_UPPER_RIGHT_CORNER: sebie->model.resize_corner = SEBIE_LOWER_LEFT_CORNER; break;
                case SEBIE_LOWER_RIGHT_CORNER: sebie->model.resize_corner = SEBIE_UPPER_LEFT_CORNER; break;
                case SEBIE_LOWER_LEFT_CORNER: sebie->model.resize_corner = SEBIE_UPPER_RIGHT_CORNER; break;
            }
        }
    } else {
        double xstart, ystart;
        gtk_gesture_drag_get_start_point(gesture, &xstart, &ystart);
        sebie->model.selection_x = xstart + xmove - sebie->model.mouse_pos_lx;
        sebie->model.selection_y = ystart + ymove - sebie->model.mouse_pos_ly;
    }

    sebie->dirty_tags |= SEBIE_UPDATE_SELECTION_RECT;
    sebie_update_view(sebie, NULL, NULL, 0, 0);

    drawingarea1_update_cursor(x, y, sebie);
}


void
on_drawingarea1_drag_end(GtkGestureDrag *gesture,
                         double x,
                         double y,
                         gpointer user_data)
{
    struct sebie_s *sebie = (struct sebie_s *)user_data;

    sebie_debug("on_drawingarea1_drag_end: x: %f y: %f\n", x, y);

    sebie->model.mouse_click_type = SEBIE_NOT_CLICKED;
}


void
on_drawingarea1_motion(GtkEventControllerMotion *controller,
                       double x,
                       double y,
                       gpointer user_data)
{
    struct sebie_s *sebie = (struct sebie_s *)user_data;

    sebie_debug("on_drawingarea1_motion: x: %f y: %f\n", x, y);

    drawingarea1_update_cursor(x, y, sebie);

    sebie->dirty_tags |= SEBIE_UPDATE_SELECTION_RECT;
    sebie_update_view(sebie, NULL, NULL, 0, 0);
}
#else
gboolean
on_drawingarea1_button_press_event (GtkWidget       *widget,
                    GdkEventButton  *event,
                    gpointer         user_data)
{
  struct sebie_s *sebie = (struct sebie_s *)user_data;

  if (event->button == 1) { /* left button */
    sebie_debug("button_press_event %f %f\n", event->x, event->y);
    if (in_upper_left_corner(event->x, event->y, sebie)) {
      sebie_debug("UPPER_LEFT\n");
      sebie->model.mouse_click_type = SEBIE_RESIZE_SELECTION;
      sebie->model.resize_corner = SEBIE_UPPER_LEFT_CORNER;
    } else if (in_upper_right_corner(event->x, event->y, sebie)) {
      sebie_debug("UPPER_RIGHT\n");
      sebie->model.mouse_click_type = SEBIE_RESIZE_SELECTION;
      sebie->model.resize_corner = SEBIE_UPPER_RIGHT_CORNER;
    } else if (in_lower_right_corner(event->x, event->y, sebie)) {
      sebie_debug("LOWER_RIGHT\n");
      sebie->model.mouse_click_type = SEBIE_RESIZE_SELECTION;
      sebie->model.resize_corner = SEBIE_LOWER_RIGHT_CORNER;
    } else if (in_lower_left_corner(event->x, event->y, sebie)) {
      sebie_debug("LOWER_LEFT\n");
      sebie->model.mouse_click_type = SEBIE_RESIZE_SELECTION;
      sebie->model.resize_corner = SEBIE_LOWER_LEFT_CORNER;
    } else if (in_selected_area(event->x, event->y, sebie)) {
      sebie_debug("MOVE\n");
      sebie->model.mouse_click_type = SEBIE_MOVE_SELECTION;
      sebie->model.mouse_pos_lx = event->x;
      sebie->model.mouse_pos_ly = event->y;
    } else {
      sebie_debug("LOWER_RIGHT NEW\n");
      sebie->model.mouse_click_type = SEBIE_RESIZE_SELECTION;
      sebie->model.resize_corner = SEBIE_LOWER_RIGHT_CORNER;
      sebie->model.selection_x = event->x;
      sebie->model.selection_y = event->y;
      sebie->model.selection_width  = 0;
      sebie->model.selection_height = 0;
    }

    sebie->dirty_tags |= SEBIE_UPDATE_SELECTION_RECT;
    sebie_update_view(sebie);
  }

  return FALSE;
}
#endif


#if !GTK_CHECK_VERSION(4,0,0)
gboolean
on_drawingarea1_motion_notify_event (GtkWidget       *widget,
                     GdkEventMotion  *event,
                     gpointer         user_data)
{
  struct sebie_s *sebie = (struct sebie_s *)user_data;

  gint x;
  gint y;
  GdkModifierType state;

#if GTK_CHECK_VERSION(3,0,0)
  GdkDisplay* display = gdk_display_get_default();
  GdkSeat* seat = gdk_display_get_default_seat(display);
  GdkDevice* pointer = gdk_seat_get_pointer(seat);
#endif

  if (event->is_hint) {
#if GTK_CHECK_VERSION(3,0,0)
    gdk_window_get_device_position(event->window, pointer, &x, &y, &state);
#else
    gdk_window_get_pointer (event->window, &x, &y, &state);
#endif
  } else {
    x = event->x;
    y = event->y;
    state = event->state;
  }

  if (state & GDK_BUTTON1_MASK) {
    sebie_debug("motion_notify_event %d %d \n", x, y);
    if (sebie->model.mouse_click_type == SEBIE_RESIZE_SELECTION) {
      sebie_debug("resize selection\n");
#if 0
      gint x1 = sebie->model.selection_x;
      gint y1 = sebie->model.selection_y;
#endif
      gint x2 = sebie->model.selection_x + sebie->model.selection_width;
      gint y2 = sebie->model.selection_y + sebie->model.selection_height;

      if (sebie->model.resize_corner == SEBIE_UPPER_LEFT_CORNER) {
        gint nwidth  = sebie->model.selection_x + sebie->model.selection_width - x;
        gint nheight = sebie->model.selection_y + sebie->model.selection_height - y;
        double nfrac = (double)nwidth / (double)nheight;
        if (nfrac >= sebie->model.selection_fac) {
          sebie->model.selection_x = x2 - (double)nheight * sebie->model.selection_fac;
          sebie->model.selection_y = y;
          sebie->model.selection_width  = x2 - sebie->model.selection_x;
          sebie->model.selection_height = y2 - sebie->model.selection_y;
        } else {
          sebie->model.selection_x = x;
          sebie->model.selection_y = y2 - (double)nwidth / sebie->model.selection_fac;
          sebie->model.selection_width = x2 - x;
          sebie->model.selection_height = y2 - sebie->model.selection_y;
        }
      } else if (sebie->model.resize_corner == SEBIE_UPPER_RIGHT_CORNER) {
        gint nwidth  = x - sebie->model.selection_x;
        gint nheight = sebie->model.selection_y + sebie->model.selection_height - y;
        double nfrac = (double)nwidth / (double)nheight;
        if (nfrac >= sebie->model.selection_fac) {
          sebie->model.selection_width = (double)nheight * sebie->model.selection_fac;
          sebie->model.selection_y = y;
          sebie->model.selection_height = y2 - sebie->model.selection_y;
        } else {
            sebie->model.selection_width = nwidth;
            sebie->model.selection_height = (double)nwidth / sebie->model.selection_fac;
            sebie->model.selection_y = y2 - sebie->model.selection_height;
        }
      } else if (sebie->model.resize_corner == SEBIE_LOWER_RIGHT_CORNER) {
        gint nwidth  = x - sebie->model.selection_x;
        gint nheight = y - sebie->model.selection_y;
        double nfrac = (double)nwidth / (double)nheight;
        if (nfrac >= sebie->model.selection_fac) {
          sebie->model.selection_width = (double)nheight * sebie->model.selection_fac;
          sebie->model.selection_height = nheight;
        } else {
          sebie->model.selection_width = nwidth;
          sebie->model.selection_height = (double)nwidth / sebie->model.selection_fac;
        }
      } else { /* SEBIE_LOWER_LEFT_CORNER */
        gint nwidth  = sebie->model.selection_x + sebie->model.selection_width - x;
        gint nheight = y - sebie->model.selection_y;
        double nfrac = (double)nwidth / (double)nheight;
        if (nfrac >= sebie->model.selection_fac) {
          sebie->model.selection_width = (double)nheight * sebie->model.selection_fac;
          sebie->model.selection_height = nheight;
          sebie->model.selection_x = x2 - sebie->model.selection_width;
        } else {
          sebie->model.selection_x = x;
          sebie->model.selection_height = (double)nwidth / sebie->model.selection_fac;
          sebie->model.selection_width = x2 - sebie->model.selection_x;
        }
      }
      /* adjust if selection crossed the fix point */
      if (sebie->model.selection_width < 0 &&
          sebie->model.selection_height < 0) {
        sebie->model.selection_x += sebie->model.selection_width;
        sebie->model.selection_width = sebie->model.selection_width * (-1);
        sebie->model.selection_y += sebie->model.selection_height;
        sebie->model.selection_height = sebie->model.selection_height * (-1);
        switch(sebie->model.resize_corner) {
          case SEBIE_UPPER_LEFT_CORNER: sebie->model.resize_corner = SEBIE_LOWER_RIGHT_CORNER; break;
          case SEBIE_UPPER_RIGHT_CORNER: sebie->model.resize_corner = SEBIE_LOWER_LEFT_CORNER; break;
          case SEBIE_LOWER_RIGHT_CORNER: sebie->model.resize_corner = SEBIE_UPPER_LEFT_CORNER; break;
          case SEBIE_LOWER_LEFT_CORNER: sebie->model.resize_corner = SEBIE_UPPER_RIGHT_CORNER; break;
        }
      }
    } else {
      sebie_debug("move selection\n");
      gint diff_x = x - sebie->model.mouse_pos_lx;
      gint diff_y = y - sebie->model.mouse_pos_ly;
      sebie->model.selection_x += diff_x;
      sebie->model.selection_y += diff_y;
      sebie->model.mouse_pos_lx = x;
      sebie->model.mouse_pos_ly = y;
    }
    sebie->dirty_tags |= SEBIE_UPDATE_SELECTION_RECT;
    sebie_update_view(sebie);
  }

  if (in_selected_area(x,y, sebie)) {
#if GTK_CHECK_VERSION(3,0,0)
    GdkCursor *cursor = gdk_cursor_new_for_display(display, GDK_FLEUR);
#else
    GdkCursor *cursor = gdk_cursor_new(GDK_FLEUR);
#endif
    gdk_window_set_cursor(event->window, cursor);
  } else if (in_upper_left_corner(x, y, sebie) ||
         ((state & GDK_BUTTON1_MASK) &&
          sebie->model.resize_corner == SEBIE_UPPER_LEFT_CORNER)) {
#if GTK_CHECK_VERSION(3,0,0)
    GdkCursor *cursor = gdk_cursor_new_for_display(display, GDK_TOP_LEFT_CORNER);
#else
    GdkCursor *cursor = gdk_cursor_new(GDK_TOP_LEFT_CORNER);
#endif
    gdk_window_set_cursor(event->window, cursor);
  } else if (in_upper_right_corner(x, y, sebie) ||
         ((state & GDK_BUTTON1_MASK) &&
          sebie->model.resize_corner == SEBIE_UPPER_RIGHT_CORNER)) {
#if GTK_CHECK_VERSION(3,0,0)
    GdkCursor *cursor = gdk_cursor_new_for_display(display, GDK_TOP_RIGHT_CORNER);
#else
    GdkCursor *cursor = gdk_cursor_new(GDK_TOP_RIGHT_CORNER);
#endif
    gdk_window_set_cursor(event->window, cursor);
  } else if (in_lower_right_corner(x, y, sebie) ||
         ((state & GDK_BUTTON1_MASK) &&
          sebie->model.resize_corner == SEBIE_LOWER_RIGHT_CORNER)) {
#if GTK_CHECK_VERSION(3,0,0)
    GdkCursor *cursor = gdk_cursor_new_for_display(display, GDK_BOTTOM_RIGHT_CORNER);
#else
    GdkCursor *cursor = gdk_cursor_new(GDK_BOTTOM_RIGHT_CORNER);
#endif
    gdk_window_set_cursor(event->window, cursor);
  } else if (in_lower_left_corner(x, y, sebie) ||
         ((state & GDK_BUTTON1_MASK) &&
          sebie->model.resize_corner == SEBIE_LOWER_LEFT_CORNER)) {
#if GTK_CHECK_VERSION(3,0,0)
    GdkCursor *cursor = gdk_cursor_new_for_display(display, GDK_BOTTOM_LEFT_CORNER);
#else
    GdkCursor *cursor = gdk_cursor_new(GDK_BOTTOM_LEFT_CORNER);
#endif
    gdk_window_set_cursor(event->window, cursor);
  } else {
    gdk_window_set_cursor (event->window, NULL);
  }

  return TRUE;
}
#endif


#if GTK_CHECK_VERSION(4,0,0)
void
on_error_dialog_response (GtkDialog *dialog,
                          int response,
                          gpointer user_data)
{
    sebie_debug("on_error_dialog_response() - response: %d\n", response);
    gtk_window_destroy (GTK_WINDOW(dialog));
}
#endif
