/**
 *  Copyright 2003-2022 Peter Seiderer <Peter.Seiderer@ciselant.de>
 *
 *  This file is part of SeBIE.
 *
 *  SeBIE is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  SeBIE is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with SeBIE; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
#include "sebie_config.h"
#include "sebie_helpers.h"

#include <gtk/gtk.h>
#include <gdk/gdk.h>

#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>


#define BUF_LEN 1024


int
sebie_read_config(const char *path, sebie_config_spec *spec)
{
  char buf[BUF_LEN];
  FILE *file;
  int line = 0;
  char *eqs;
  char *name;
  char *value;
  char *tmp;
  int i = 0;
  
  /* init with default values */
  while (spec[i].type != CONFIG_END) {
    switch (spec[i].type) {
    case CONFIG_INTEGER:
      *spec[i].storage.integer = spec[i].defvalue.integer;
      break;
    case CONFIG_DOUBLE:
      *spec[i].storage.doublev = spec[i].defvalue.doublev;
      break;
    case CONFIG_STRING:
      *spec[i].storage.string = g_strdup(spec[i].defvalue.string);
      break;
    default:
      sebie_error("Error: unknown config type %d.\n", spec[i].type);
      return 1;
    }
    i++;
  }

  if ((file = fopen(path, "r")) == NULL) {
    sebie_debug("open of config file %s failed (%s).\n",
		path,
		strerror(errno));
    return 1;
  }
  
  while (fgets(buf, BUF_LEN, file) != NULL) {
    line++;
    if (buf[0] == '#' ||                       // comment line
	(eqs = strchr(buf, '=')) == NULL ||    // no equal sign
	eqs == buf) {                          // line starts with the equal sign
      // skip
      continue;
    }
    
    if ((tmp = strchr(buf, '\n')) != NULL) {
      *tmp = '\0';
    }
    
    *eqs = '\0';

    /* eat spaces */
    tmp =  eqs -1;
    while (tmp >= buf) {
      if (*tmp == ' ') {
	*tmp = '\0';
	tmp--;
	continue;
      }
      break;
    }
    name = buf;

    /* eat spaces */
    tmp =  eqs +1;
    while (*tmp != '\0') {
      if (*tmp == ' ') {
	*tmp = '\0';
	tmp++;
	continue;
      }
      break;
    }
    value = tmp;
    
    i = 0;
    while (spec[i].type != CONFIG_END) {
      if (!strcmp(spec[i].name, name)) {
	switch (spec[i].type) {
	case CONFIG_INTEGER:
	  *spec[i].storage.integer = atoi(value);
	  break;
	case CONFIG_DOUBLE:
	  *spec[i].storage.doublev = atof(value);
	  break;
	case CONFIG_STRING:
	  if (*spec[i].storage.string != NULL) {
	    g_free(*spec[i].storage.string);
	  }
	  *spec[i].storage.string = g_strdup(value);
	  break;
	default:
	  sebie_error("Error: unknown config type %d.\n", spec[i].type);
	  fclose(file);
	  return 1;
	}
      }
      i++;
    }
  }

  fclose(file);
  
  return 0;
}


int
sebie_write_config(const char *path, sebie_config_spec *spec)
{
  int i = 0;
  FILE *file;
  
  if ((file = fopen(path, "w")) == NULL) {
    fprintf(stderr, "Error: open of config file %s failed (%s).\n",
	    path, strerror(errno));
    return 1;
  }
  
  while (spec[i].type != CONFIG_END) {
    switch (spec[i].type) {
    case CONFIG_INTEGER:
      fprintf(file, "%s=%d\n", spec[i].name, *spec[i].storage.integer);
      break;
    case CONFIG_DOUBLE:
      fprintf(file, "%s=%lf\n", spec[i].name, *spec[i].storage.doublev);
      break;
    case CONFIG_STRING:
      fprintf(file, "%s=%s\n", spec[i].name, *spec[i].storage.string);
      break;
    default:
	/* write out error and ignore */
      sebie_error("Error: unknown config type %d.\n", spec[i].type);
    }
    i++;
  }

  fclose(file);

  return 0;
}
