/**
 *  Copyright 2002 Peter Seiderer <Peter.Seiderer@ciselant.de>
 *
 *  This file is part of SeBIE.
 *
 *  SeBIE is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  SeBIE is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with SeBIE; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
#include "sebie.h"
#include "sebie_helpers.h"
#include "sebie_view.h"
#include "sebie_gtk_helpers.h"
#include "sebie_config.h"

#include <gdk/gdkkeysyms.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>


void
sebie_update_view(struct sebie_s *sebie)
{
  if (sebie->dirty_tags & SEBIE_UPDATE_SCALED_PIXBUF) {
    GdkGC *gdkGC = gdk_gc_new(sebie->view.drawingarea1->window);
    GdkColor color;
    color.red   = 0xe0e0;
    color.green = 0xe0e0;
    color.blue  = 0xe0e0;
    gdk_gc_set_rgb_fg_color(gdkGC, &color);
    gdk_draw_rectangle(sebie->view.drawingarea1->window,
		       gdkGC,
		       TRUE,
		       0, 0,
		       -1, -1);
    if (sebie->model.scaled_pixbuf != NULL) {
      gdk_pixbuf_render_to_drawable(sebie->model.scaled_pixbuf,
				    sebie->view.drawingarea1->window,
				    gdkGC,
				    0, 0,
				    0, 0,
				    sebie->model.scaled_width,
				    sebie->model.scaled_height,
				    GDK_RGB_DITHER_NONE,
				    0,
				    0);
    }
  }
  
  if (sebie->dirty_tags & SEBIE_UPDATE_SELECTION_FAC) {
    gchar buf[6];
    snprintf(buf, 6, "%5d", sebie->model.selection_fac_width);
    gtk_entry_set_text(GTK_ENTRY(sebie->view.entry_selection_fac_width), buf);
    snprintf(buf, 6, "%5d", sebie->model.selection_fac_height);
    gtk_entry_set_text(GTK_ENTRY(sebie->view.entry_selection_fac_height), buf);
    sebie->model.selection_fac = (double)sebie->model.selection_fac_width / (double) sebie->model.selection_fac_height;
    snprintf(buf, 6, "%5.2f", sebie->model.selection_fac);
    gtk_label_set_text(GTK_LABEL(sebie->view.label_selection_fac), buf);
  }

  if (sebie->dirty_tags & SEBIE_UPDATE_SELECTION_RECT) {
    GdkGC *gdkGC = gdk_gc_new(sebie->view.drawingarea1->window);
    if (sebie->model.scaled_pixbuf != NULL) {
      gdk_pixbuf_render_to_drawable (sebie->model.scaled_pixbuf,
				     sebie->view.drawingarea1->window,
				     gdkGC,
				     0, 0,
				     0, 0,
				     sebie->model.scaled_width,
				     sebie->model.scaled_height,
				     GDK_RGB_DITHER_NONE,
				     0,
				     0);
    }
    
    gint rx, ry, rheight, rwidth;

    if (sebie->model.selection_width > 0) {
      rx = sebie->model.selection_x;
      rwidth = sebie->model.selection_width;
    } else {
      rx = sebie->model.selection_x + sebie->model.selection_width;
      rwidth = - sebie->model.selection_width;
    }
    
    if (sebie->model.selection_height > 0) {
      ry = sebie->model.selection_y;
      rheight = sebie->model.selection_height;
    } else {
      ry = sebie->model.selection_y + sebie->model.selection_height;
      rheight = - sebie->model.selection_height;
    }

    gdk_gc_set_function(gdkGC, GDK_INVERT);
    gdk_draw_rectangle (sebie->view.drawingarea1->window,
			gdkGC,
			FALSE,
			rx,
			ry,
			rwidth,
			rheight);
  }

  if (sebie->dirty_tags & SEBIE_UPDATE_INPUT_PATH) {
    gtk_entry_set_text(GTK_ENTRY(sebie->view.entry_input_path),
		       sebie->model.input_path);
  }
  if (sebie->dirty_tags & SEBIE_UPDATE_INPUT_REGEX) {
    gtk_entry_set_text(GTK_ENTRY(sebie->view.entry_input_regex),
		       sebie->model.input_regex);
  }
  if (sebie->dirty_tags & SEBIE_UPDATE_OUTPUT_FORMAT) {
    gtk_entry_set_text(GTK_ENTRY(sebie->view.entry_output_format),
		       sebie->model.output_format);
  }
  
  if (sebie->dirty_tags & SEBIE_UPDATE_OUTPUT_FILE_NAME) {
    gtk_entry_set_text(GTK_ENTRY(sebie->view.entry_output_file),
		       sebie->model.output_file_name);
  }

  if (sebie->dirty_tags & SEBIE_UPDATE_STATUS_BAR) {
    gtk_statusbar_pop(GTK_STATUSBAR(sebie->view.statusbar1), sebie->view.statusbar1_context_id);
    gtk_statusbar_push(GTK_STATUSBAR(sebie->view.statusbar1), sebie->view.statusbar1_context_id, sebie->model.statusbar_text);
  }

  sebie->dirty_tags = 0;
  save_sebie_config(sebie);
}




void
sebie_file_list_clear(struct sebie_s *sebie)
{
  if (sebie->model.file_list == NULL) {
    return;
  }
  gtk_list_store_clear(sebie->model.file_list);
}


void
sebie_file_list_add(struct sebie_s *sebie, G_CONST_RETURN gchar *filename)
{
  if (sebie->model.file_list == NULL) {
    sebie->model.file_list = gtk_list_store_new(1, G_TYPE_STRING);
    gtk_tree_view_set_model(GTK_TREE_VIEW(sebie->view.treeview_file_list), GTK_TREE_MODEL(sebie->model.file_list));
  }
  GtkTreeIter iter;
  gtk_list_store_append(sebie->model.file_list, &iter);
  gtk_list_store_set(sebie->model.file_list, &iter, 0, filename, -1);
}

void
load_pixbuf (struct sebie_s *sebie)
{
  if (sebie->model.input_file_name == NULL) {
    return;
  }
  
  GError *error = NULL;
  GdkPixbuf *pixbuf;

  gchar pathname[PATH_MAX];
  g_snprintf(pathname, PATH_MAX, "%s/%s", sebie->model.input_path, sebie->model.input_file_name);
  gchar statusbar_text[PATH_MAX + 64];
  g_snprintf(statusbar_text, PATH_MAX + 64, "   Load file: '%s'", pathname);
  sebie_set_string(&sebie->model.statusbar_text, statusbar_text);
  sebie->dirty_tags |= SEBIE_UPDATE_STATUS_BAR;

  if ((pixbuf = gdk_pixbuf_new_from_file(pathname, &error)) == NULL) {
    g_print("gdk_pixbuf_new_from_file failed (%s).\n", error->message);
    return;
  }
  sebie->model.image_width = gdk_pixbuf_get_width(pixbuf);
  sebie->model.image_height = gdk_pixbuf_get_height(pixbuf);
  if (sebie->model.image_pixbuf != NULL) {
    g_object_unref(sebie->model.image_pixbuf);
  }
  sebie->model.image_pixbuf = pixbuf;
  double scale1 = 1024.0 / (double)sebie->model.image_width;
  double scale2 =  768.0 / (double)sebie->model.image_width;
  double scale3 = 1024.0 / (double)sebie->model.image_height;
  double scale4 =  768.0 / (double)sebie->model.image_height;
    
  scale1 = MIN(scale1, scale2);
  scale1 = MIN(scale1, scale3);
  scale1 = MIN(scale1, scale4);
  printf("scale: %f \n", scale1);
  sebie->model.scaled_scale = scale1;
  sebie->model.scaled_width  = (int)((double)sebie->model.image_width * scale1);
  sebie->model.scaled_height = (int)((double)sebie->model.image_height * scale1);
  sebie->model.scaled_rotation = SEBIE_ROTATE_000;

  printf("scaled: width: %d height: %d \n",
	 sebie->model.scaled_width, 
	 sebie->model.scaled_height);
  if (sebie->model.scaled_pixbuf != NULL) {
    g_object_unref(sebie->model.scaled_pixbuf);
  }
  sebie->model.scaled_pixbuf = gdk_pixbuf_scale_simple(sebie->model.image_pixbuf,
						       sebie->model.scaled_width,
						       sebie->model.scaled_height,
						       GDK_INTERP_HYPER);
  sebie->dirty_tags |= SEBIE_UPDATE_SCALED_PIXBUF;
#if 0
  sebie_update_view(sebie);
#endif
}


void
sebie_update_file_list(struct sebie_s *sebie)
{
  GError *error = NULL;
  GDir *gdir;
  save_sebie_config(sebie);
  sebie_file_list_clear(sebie);
  if ((gdir = g_dir_open(sebie->model.input_path, 0, &error)) == NULL) {
    return;
  }

  G_CONST_RETURN gchar *next;
  while((next = g_dir_read_name(gdir)) != NULL) {
    printf("next %s\n", next);

    if (regexec(&sebie->model.regex, next, 0, NULL, 0) == 0) {
      sebie_file_list_add(sebie, next);
    }
  }
}


void
rotate_scaled(struct sebie_s *sebie, int direction)
{
  if (direction == SEBIE_ROTATE_090 ||
      direction == SEBIE_ROTATE_270 ) {
    GdkPixbuf *rotated = rotate_pixbuf(sebie->model.scaled_pixbuf, direction);
    
    sebie->model.scaled_width = gdk_pixbuf_get_width(rotated);
    sebie->model.scaled_height = gdk_pixbuf_get_height(rotated);
    if (sebie->model.scaled_pixbuf != NULL) {
      g_object_unref(sebie->model.scaled_pixbuf);
    }
    sebie->model.scaled_pixbuf = rotated;
  
    sebie->dirty_tags |= SEBIE_UPDATE_SCALED_PIXBUF;
    sebie_update_view(sebie);
  }
}


void
save_sebie_config(struct sebie_s *sebie)
{
  if (sebie->model_tags) {
    sebie_config_spec config_spec[] = {
      { "InputPath",  CONFIG_STRING, &sebie->model.input_path, "" },
      { "InputRegEx", CONFIG_STRING, &sebie->model.input_regex, "\\([0-9]*\\).jpg" },
      { "OutputFormat", CONFIG_STRING, &sebie->model.output_format, "Image%s.jpg" },
      { NULL, CONFIG_END, NULL, 0}
    };
  
    sebie_write_config(sebie->config_file, config_spec);
    sebie->model_tags = 0;
  }
}


void
init_sebie_model(struct sebie_s *sebie)
{
  char *homedir;
  char buf[PATH_MAX];
  
  if ((homedir = getenv("HOME")) != NULL) {
    snprintf(buf, PATH_MAX, "%s/%s", homedir, ".sebieconf");
    sebie->config_file = g_strdup(buf);
  } else {
    sebie->config_file = g_strdup(".sebieconf");
  }
  
  sebie_config_spec config_spec[] = {
    { "InputPath",  CONFIG_STRING, &sebie->model.input_path, "" },
    { "InputRegEx", CONFIG_STRING, &sebie->model.input_regex, "\\([0-9]*\\).jpg" },
    { "OutputFormat", CONFIG_STRING, &sebie->model.output_format, "Image%s.jpg" },
    { NULL, CONFIG_END, NULL, 0}
  };
  
  sebie_read_config(sebie->config_file, config_spec);

  sebie->model.input_file_path = NULL;
  sebie->model.output_file_path = NULL;

  sebie->dirty_tags |= SEBIE_UPDATE_INPUT_PATH;
  sebie->dirty_tags |= SEBIE_UPDATE_INPUT_REGEX;
  sebie->dirty_tags |= SEBIE_UPDATE_OUTPUT_FORMAT;

  sebie->model.file_list = gtk_list_store_new(1, G_TYPE_STRING);
  
  sebie->model.selection_fac_width = 1024;
  sebie->model.selection_fac_height = 768;
  sebie->model.selection_fac = 1024.0 / 768.0;

  sebie->model.mouse_click_type = SEBIE_NOT_CLICKED;
  
  sebie->dirty_tags |= SEBIE_UPDATE_SELECTION_FAC;

  sebie->model.statusbar_text = strdup("   SeBIE  (C) 2003 by Peter.Seiderer@ciselant.de");
  sebie->dirty_tags |= SEBIE_UPDATE_STATUS_BAR;
}


int main(int argc, char *argv[])
{
  GtkWidget *window1;
  struct sebie_s *sebie;
  
  sebie = g_new(struct sebie_s, 1);

  gtk_set_locale ();

  gtk_init (&argc, &argv);
  
  init_sebie_model(sebie);
  window1 = init_sebie_view(sebie);
  sebie_update_view(sebie);

  gtk_widget_show (window1);

  gtk_main ();

  exit(0);
}
