/*
 * libci_diff, a library to compute a binary copy/insert diff.
 * Copyright (C) 2001 Peter Seiderer <Peter.Seiderer@ciselant.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifndef _CI_DIFF_H_
#define _CI_DIFF_H_ 1

#include <sys/types.h>

#ifdef __cplusplus
extern "C" {
#endif


/*
 * The function computes the difference between memory areas s1 and s2.
 * 
 * The result is a binary encoding of the corresponding copy/insert
 * instructions to convert the content of memory areas s1 to the content
 * of memory area s2.
 * The result is stored in new allocated memory area pointed to by delta and
 * the length of the output is stored in the variable pointed to by delta_len.
 *
 * returns: 0 in case of success, errno in case of failure.
 */
int ci_diff_b( void **delta,      /* pointer to output buffer */
	       size_t *delta_len, /* pointer to output buffer length */
	       void *s1,          /* input buffer s1 */
	       size_t n1,         /* length of input buffer s1 */
	       void *s2,          /* input buffer s2 */
	       size_t n2          /* length of input buffer s2 */
	       );
	
 /*
 * The function computes the difference between the strings s1 and s2.
 * 
 * The result is a string encoding of the corresponding copy/insert
 * instructions to convert the content of the string s1 to the content
 * of the string s2.
 * The result is stored in new allocated memory area pointed to by delta.
 *
 * returns: 0 in case of success, errno in case of failure.
 */
int ci_diff_s( char **delta,      /* pointer to output buffer */
	       char *s1,          /* input buffer s1 */
	       char *s2           /* input buffer s2 */
	       );

/*
 * The function computes the difference between the strings s1 and s2.
 * 
 * The result is a string encoding of the corresponding copy/insert
 * instructions to convert the content of the string s1 of lenght n1
 * to the content of the string s2 of length n2.
 * The result is stored in new allocated memory area pointed to by delta and
 * the length of the output is stored in the variable pointed to by delta_len.
 *
 * returns: 0 in case of success, errno in case of failure.
 */
int ci_diff_sn( char **delta,      /* pointer to output buffer */
		size_t *delta_len, /* pointer to output buffer length */
		char *s1,          /* input buffer s1 */
		size_t n1,         /* lenght of string s1 */
		char *s2,          /* input buffer s2 */
		size_t n2          /* length of string s2 */
		);

/*
 * The function applies the delta sequence to input s2.
 * 
 * The function applies the binary encoded delta sequence of copy/insert
 * instructions in the memory area pointed to by delta wiht length
 * delta_len to the input memory area s2.
 * 
 * The result is stored in new allocated memory area pointed to by s1 and
 * the length of the output is stored in the variable pointed to by n1.
 *
 * returns: 0 in case of success, errno in case of failure.
 */
int ci_apply_b( void **s1,        /* pointer to output buffer s1*/
		size_t *n1,       /* pointer to output buffer length */
		void *s2,         /* input buffer s2 */
		size_t n2,        /* length of input buffer s2 */
		void *delta,      /* input buffer delta */
		size_t delta_len  /* length of input buffer delta */
		);
	
/*
 * The function applies the delta sequence to input s2.
 * 
 * The function applies the string encoded delta sequence of copy/insert
 * instructions from the string pointed to by delta to the input
 * string  s2.
 * 
 * The result is stored in new allocated memory area pointed to by s1.
 *
 * returns: 0 in case of success, errno in case of failure.
 */
int ci_apply_s( char **s1,      /* pointer to output buffer s1 */
		char *s2,       /* input buffer s2 */
		char *delta     /* input buffer delta */
		);

/*
 * The function applies the delta sequence to input s2.
 * 
 * The function applies the string encoded delta sequence of copy/insert
 * instructions from the string pointed to by delta to the input
 * string  s2.
 * 
 * The result is stored in new allocated memory area pointed to by s1.
 *
 * returns: 0 in case of success, errno in case of failure.
 */
int ci_apply_sn( char **s1,       /* pointer to output buffer s1 */
		 size_t *n1,      /* pointer to output buffer length */
		 char *s2,        /* input buffer s2 */
		 size_t n2,       /* length of input buffer s2 */
		 char *delta,     /* input buffer delta */
		 size_t delta_len /* length of input buffer delta */
		 );
  
#ifdef __cplusplus
}
#endif
      
#endif
