/**
 *  Copyright 2002 Peter Seiderer <Peter.Seiderer@ciselant.de>
 *
 *  This file is part of Jardin.
 *
 *  Jardin is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Jardin is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Foobar; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */
#include "jardin_cap.h"
#include "jardin_debug.h"

#include <stdio.h>
#include <sys/param.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <sys/mman.h>

static char *jardin_cap_files[] = {
  "Header.cap",
  "Directory.cap",
  "Import.cap",
  "Descriptor.cap",
  "Applet.cap",
  "Class.cap",
  "Method.cap",
  "StaticField.cap",
  "Export.cap",
  "ConstantPool.cap",
  "ReferenceLocation",
  //"Reflection.cap",
  NULL
};

static char *jardin_component_types[] = {
  "unknown",
  "Header",
  "Directory",
  "Applet",
  "Import",
  "ConstantPool",
  "Class",
  "Method",
  "StaticField",
  "ReferenceLocation",
  "Export",
  "Descriptor"
};

static int method_count;


static u2 find_bytecode_count(struct jardin_cap *cap, off_t index, struct method_info *method_info) {
  int i, j;
  for (i = 0; i < cap->descriptor_comp.class_count; i++) {
    for (j = 0; j < cap->descriptor_comp.classes[i].method_count; j++) {
      if (cap->descriptor_comp.classes[i].methods[j].method_offset == index) {
	cap->descriptor_comp.classes[i].methods[j].method_info = method_info;
	return cap->descriptor_comp.classes[i].methods[j].bytecode_count;
      }
    }
  }
  JARDIN_FATAL("no method with this offset (0x%lx) found", index);
}


#define U1(a) if(*index>=index_max)return(-1);printf("0x%.04lx u1:  ",*index);a=cap_file_mmap[*index];(*index)+=1;printf(#a": 0x%.02x\n",a);
#define U2(a) if(*index>=(index_max-1))return(-1);;printf("0x%.04lx u2:  ",*index);a=cap_file_mmap[*index]*256+cap_file_mmap[(*index)+1];(*index)+=2;printf(#a": 0x%.04x\n",a);

static int jardin_cap_fill(struct jardin_cap *cap, unsigned char *cap_file_mmap, off_t *index, off_t index_max) {
  int i, j, k;
  //off_t mmap_index;
  //off_t length_index;
  u2 bytecode_count;

  switch (cap_file_mmap[0]) {
  case COMPONENT_Header:    
    U1(cap->header_comp.tag);
    U2(cap->header_comp.size);
    U1(cap->header_comp.magic[0]);
    U1(cap->header_comp.magic[1]);
    U1(cap->header_comp.magic[2]);
    U1(cap->header_comp.magic[3]);
    U1(cap->header_comp.minor_version);
    U1(cap->header_comp.major_version);
    U1(cap->header_comp.flags);
    U1(cap->header_comp.this_package.minor_version);
    U1(cap->header_comp.this_package.major_version);
    U1(cap->header_comp.this_package.AID_length);
    cap->header_comp.this_package.AID = calloc(1, cap->header_comp.this_package.AID_length);
    for (i = 0; i < cap->header_comp.this_package.AID_length; i++) {
      U1(cap->header_comp.this_package.AID[i]);
    }
    break;
  case COMPONENT_Directory:
    U1(cap->directory_comp.tag);
    U2(cap->directory_comp.size);;
    for (i = 0; i < 11; i++) {
      U2(cap->directory_comp.component_sizes[i])
    }
    U2(cap->directory_comp.static_field_size.image_size);
    U2(cap->directory_comp.static_field_size.array_init_count);
    U2(cap->directory_comp.static_field_size.array_init_size);
    U1(cap->directory_comp.import_count);
    U1(cap->directory_comp.applet_count);
    U1(cap->directory_comp.custom_count);
    if (cap->directory_comp.custom_count > 0) {
      cap->directory_comp.custom_components = calloc(cap->directory_comp.custom_count, sizeof(struct custom_component_info));
      for (i = 0; i < cap->directory_comp.custom_count; i++) {
	U1(cap->directory_comp.custom_components[i].component_tag);
	U2(cap->directory_comp.custom_components[i].size);
	U1(cap->directory_comp.custom_components[i].AID_length);
	cap->directory_comp.custom_components[i].AID = calloc(1, cap->directory_comp.custom_components[i].AID_length);
	for (j = 0; j < cap->directory_comp.custom_components[i].size; j++) {
	  U1(cap->directory_comp.custom_components[i].AID[j]);
	}
      }
    }
    break;
  case COMPONENT_Applet:
    U1(cap->applet_comp.tag);
    U2(cap->applet_comp.size);
    U1(cap->applet_comp.count);
    if (cap->applet_comp.count > 0) {
      cap->applet_comp.applets = calloc(cap->applet_comp.count, sizeof(struct applets));
      for (i = 0; i < cap->applet_comp.count; i++) {
	U1(cap->applet_comp.applets[i].AID_length);
	cap->applet_comp.applets[i].AID = calloc(1, cap->applet_comp.applets[i].AID_length);
	for (j = 0; j < cap->applet_comp.applets[i].AID_length; j++) {
	  U1(cap->applet_comp.applets[i].AID[j]);
	}
	U2(cap->applet_comp.applets[i].install_method_offset);
      }
    }
    break;
  case COMPONENT_Import:
    U1(cap->import_comp.tag);
    U2(cap->import_comp.size);
    U1(cap->import_comp.count);
    if (cap->import_comp.count > 0) {
      cap->import_comp.packages = calloc(cap->import_comp.count, sizeof(struct package_info));
      for (i = 0; i < cap->import_comp.count; i++) {
	U1(cap->import_comp.packages[i].minor_version);
	U1(cap->import_comp.packages[i].major_version);
	U1(cap->import_comp.packages[i].AID_length);
	cap->import_comp.packages[i].AID = calloc(1, cap->import_comp.packages[i].AID_length);
	for (j = 0; j < cap->import_comp.packages[i].AID_length; j++) {
	  U1(cap->import_comp.packages[i].AID[j]);
	}
      }
    }
    break;
  case COMPONENT_ConstantPool:
    U1(cap->constant_pool_comp.tag);
    U2(cap->constant_pool_comp.size);
    U2(cap->constant_pool_comp.count);
    if (cap->constant_pool_comp.count > 0) {
      cap->constant_pool_comp.constant_pool = calloc(cap->constant_pool_comp.count, sizeof(struct cp_info) * cap->constant_pool_comp.count);
      for (i = 0; i < cap->constant_pool_comp.count; i++) {
	U1(cap->constant_pool_comp.constant_pool[i].tag);
	U1(cap->constant_pool_comp.constant_pool[i].info[0]);
	U1(cap->constant_pool_comp.constant_pool[i].info[1]);
	U1(cap->constant_pool_comp.constant_pool[i].info[2]);
      }
    }
    break;
  case COMPONENT_Class:
    {
      int interface_index = 0;
      int class_index = 0;
      u1 bitfield;
      U1(cap->class_comp.tag);
      U2(cap->class_comp.size);
      
      while(*index < (cap->class_comp.size+2)) {
	U1(bitfield);
	if (bitfield & 0x80) {
	  // interface
	  printf(" bitfield indicates interface[%d]:\n", interface_index);
	  if ((cap->class_comp.interfaces = realloc(cap->class_comp.interfaces, (interface_index + 1) * sizeof(struct interface_info))) == NULL) {
	    JARDIN_FATAL("failed to realloc struct interface_info");
	  }
	  memset(&cap->class_comp.interfaces[interface_index], 0, sizeof(struct interface_info));
	  cap->class_comp.interfaces[interface_index].bitfield = bitfield;
	  cap->class_comp.interfaces[interface_index].superinterfaces = calloc(INTERFACE_INFO_INTERFACE_COUNT(cap->class_comp.interfaces[interface_index]), sizeof(union class_ref));
	  for(j = 0; j < INTERFACE_INFO_INTERFACE_COUNT(cap->class_comp.interfaces[interface_index]); j++) {
	    U2(cap->class_comp.interfaces[interface_index].superinterfaces[j].internal_class_ref);
	  }
	  interface_index++;
	  cap->class_comp.interface_count = interface_index;
	} else {
	  // class
	  printf(" bitfield indicates class[%d]:\n", class_index);
	  printf("            cap->class_comp.clases[%d]\n", class_index);
	  if ((cap->class_comp.classes = realloc(cap->class_comp.classes, (class_index + 1) * sizeof(struct class_info))) == NULL) {
	    JARDIN_FATAL("failed to realloc struct class_info");
	  }
	  memset(&cap->class_comp.classes[class_index], 0, sizeof(struct class_info));
	  cap->class_comp.classes[class_index].cap_file_offset = (*index)-1;
	  printf("class_comp.classes[%d].cap_file_offset: 0x%.4lx\n", class_index, (*index)-1);
	  cap->class_comp.classes[class_index].bitfield = bitfield;
	  U2(cap->class_comp.classes[class_index].super_class_ref.internal_class_ref);
	  U1(cap->class_comp.classes[class_index].declared_instance_size);
	  U1(cap->class_comp.classes[class_index].first_reference_token);
	  U1(cap->class_comp.classes[class_index].reference_count);
	  U1(cap->class_comp.classes[class_index].public_method_table_base);
	  U1(cap->class_comp.classes[class_index].public_method_table_count);
	  U1(cap->class_comp.classes[class_index].package_method_table_base);
	  U1(cap->class_comp.classes[class_index].package_method_table_count);
	  cap->class_comp.classes[class_index].public_virtual_method_table = calloc(cap->class_comp.classes[class_index].public_method_table_count, sizeof(u2));
	  for (j = 0; j < cap->class_comp.classes[class_index].public_method_table_count; j++) {
	    U2(cap->class_comp.classes[class_index].public_virtual_method_table[j]);
	  }
	  cap->class_comp.classes[class_index].package_virtual_method_table = calloc(cap->class_comp.classes[class_index].package_method_table_count, sizeof(u2));
	  for (j = 0; j < cap->class_comp.classes[class_index].package_method_table_count; j++) {
	    U2(cap->class_comp.classes[class_index].package_virtual_method_table[j]);
	  }
	  cap->class_comp.classes[class_index].interfaces = calloc(CLASS_INFO_INTERFACE_COUNT(cap->class_comp.classes[class_index]), sizeof(struct implemented_interface_info));
	  for (j = 0; j < CLASS_INFO_INTERFACE_COUNT(cap->class_comp.classes[class_index]); j++) {
	    U2(cap->class_comp.classes[class_index].interfaces[j].interface.internal_class_ref);
	    U1(cap->class_comp.classes[class_index].interfaces[j].count);
	    cap->class_comp.classes[class_index].interfaces[j].index = calloc(cap->class_comp.classes[class_index].interfaces[j].count, sizeof(u1) * cap->class_comp.classes[class_index].interfaces[j].count);
	    for (k = 0; k < cap->class_comp.classes[class_index].interfaces[j].count; k++) {
	      U1(cap->class_comp.classes[class_index].interfaces[j].index[k]);
	    }
	  }
	  class_index++;
	  cap->class_comp.class_count = class_index;
	}
      }
    }
    break;
  case COMPONENT_Method:
    U1(cap->method_comp.tag);
    U2(cap->method_comp.size);
    U1(cap->method_comp.handler_count);
    cap->method_comp.exception_handlers = calloc(cap->method_comp.handler_count, sizeof(struct exception_handler_info));
    for (i = 0; i < cap->method_comp.handler_count; i++) {
      U2(cap->method_comp.exception_handlers[i].start_offset);
      U2(cap->method_comp.exception_handlers[i].bitfield);
      U2(cap->method_comp.exception_handlers[i].handler_offset);
      U2(cap->method_comp.exception_handlers[i].catch_type_index);
    }
    cap->method_comp.method_count = method_count;
    cap->method_comp.methods = calloc(method_count, sizeof(struct method_info));
    for (i = 0; i < method_count; i++) {
      cap->method_comp.methods[i].cap_file_offset = (*index) -3; // global header offset
      printf("method_comp.methods[%d].cap_file_offset: 0x%.04lx\n", i, (*index));
      U1(cap->method_comp.methods[i].method_header.bitfield1);
      if (METHOD_HEADER_INFO_FLAGS(cap->method_comp.methods[i].method_header) == ACC_EXTENDED) {
	fprintf(stderr, "Error: extended_header_info not yet implemented.\n");
	return -1;
      }
      U1(cap->method_comp.methods[i].method_header.bitfield2);
      bytecode_count = find_bytecode_count(cap, (*index)-2-3,&cap->method_comp.methods[i]); // 2 for header 3 for global header
      cap->method_comp.methods[i].bytecodes = calloc(bytecode_count, sizeof(u1));
      for (j = 0; j < bytecode_count; j ++) {
	U1(cap->method_comp.methods[i].bytecodes[j]);
      }
    }
    break;
  case COMPONENT_StaticField:
    U1(cap->static_field_comp.tag);
    U2(cap->static_field_comp.size);
    U2(cap->static_field_comp.image_size);
    U2(cap->static_field_comp.reference_count);
    U2(cap->static_field_comp.array_init_count);
    cap->static_field_comp.array_init = calloc(cap->static_field_comp.array_init_count, sizeof(struct array_init_info));
    for (i = 0; i < cap->static_field_comp.array_init_count; i++) {
      U1(cap->static_field_comp.array_init[i].type);
      U2(cap->static_field_comp.array_init[i].count);
      cap->static_field_comp.array_init[i].values = calloc(cap->static_field_comp.array_init[i].count, sizeof(u1));
      for (j = 0; j < cap->static_field_comp.array_init[i].count; j++) {
	U1(cap->static_field_comp.array_init[i].values[j]);
      }
    }
    U2(cap->static_field_comp.default_value_count);
    U2(cap->static_field_comp.non_default_value_count);
    cap->static_field_comp.non_default_values = calloc(cap->static_field_comp.non_default_value_count, sizeof(u1));
    for (i = 0; i < cap->static_field_comp.non_default_value_count; i++) {
      U1(cap->static_field_comp.non_default_values[i]);
    }
    break;
  case COMPONENT_ReferenceLocation:
    U1(cap->ref_location_comp.tag);
    U2(cap->ref_location_comp.size);
    U2(cap->ref_location_comp.byte_index_count);
    cap->ref_location_comp.offsets_to_byte_indices = calloc(cap->ref_location_comp.byte_index_count, sizeof(u1));
    for (i = 0; i < cap->ref_location_comp.byte_index_count; i++) {
      U1(cap->ref_location_comp.offsets_to_byte_indices[i]);
    }
    U2(cap->ref_location_comp.byte2_index_count);
    cap->ref_location_comp.offsets_to_byte2_indices = calloc(cap->ref_location_comp.byte2_index_count, sizeof(u1));
    for (i = 0; i < cap->ref_location_comp.byte2_index_count; i++) {
      U1(cap->ref_location_comp.offsets_to_byte2_indices[i]);
    }
    break;
  case COMPONENT_Export:
    U1(cap->export_comp.tag);
    U2(cap->export_comp.size);
    U1(cap->export_comp.class_count);
    cap->export_comp.class_exports = calloc(cap->export_comp.class_count, sizeof(struct class_export_info));
    for (i = 0; i < cap->export_comp.class_count; i++) {
      U2(cap->export_comp.class_exports[i].class_offset);
      U1(cap->export_comp.class_exports[i].static_field_count);
      U1(cap->export_comp.class_exports[i].static_method_count);
      cap->export_comp.class_exports[i].static_field_offsets = calloc(cap->export_comp.class_exports[i].static_field_count, sizeof(u2));
      for (j = 0; j < cap->export_comp.class_exports[i].static_field_count; j++) {
	U2(cap->export_comp.class_exports[i].static_field_offsets[j]);
      }
      cap->export_comp.class_exports[i].static_method_offsets = calloc(cap->export_comp.class_exports[i].static_method_count, sizeof(u2) * cap->export_comp.class_exports[i].static_method_count);
      for (j = 0; j < cap->export_comp.class_exports[i].static_method_count; j++) {
	U2(cap->export_comp.class_exports[i].static_method_offsets[j]);
      }
    }
    break;
  case COMPONENT_Descriptor:
    U1(cap->descriptor_comp.tag);
    U2(cap->descriptor_comp.size);
    U1(cap->descriptor_comp.class_count);
    cap->descriptor_comp.classes = calloc(cap->descriptor_comp.class_count, sizeof(struct class_descriptor_info));
    for (i = 0; i < cap->descriptor_comp.class_count; i++) {
      printf("  cap->descriptor_comp.classes[%i]:\n", i);
      U1(cap->descriptor_comp.classes[i].token);
      U1(cap->descriptor_comp.classes[i].access_flags);
      U2(cap->descriptor_comp.classes[i].this_class_ref.internal_class_ref);
      U1(cap->descriptor_comp.classes[i].interface_count);
      //interface_count += cap->descriptor_comp.classes[i].interface_count;
      U2(cap->descriptor_comp.classes[i].field_count);
      U2(cap->descriptor_comp.classes[i].method_count);
      method_count += cap->descriptor_comp.classes[i].method_count;
      if (cap->descriptor_comp.classes[i].interface_count > 0) {
	cap->descriptor_comp.classes[i].interfaces = calloc(cap->descriptor_comp.classes[i].interface_count, sizeof(union class_ref));
	for (j = 0; j < cap->descriptor_comp.classes[i].interface_count; j++) {
	  U2(cap->descriptor_comp.classes[i].interfaces[j].internal_class_ref);
	}
      }
      if (cap->descriptor_comp.classes[i].field_count > 0) {
	cap->descriptor_comp.classes[i].fields = calloc(cap->descriptor_comp.classes[i].field_count, sizeof(struct field_descriptor_info));
	for (j = 0; j < cap->descriptor_comp.classes[i].field_count; j++) {
	  U1(cap->descriptor_comp.classes[i].fields[j].token);
	  U1(cap->descriptor_comp.classes[i].fields[j].access_flags);
	  U2(cap->descriptor_comp.classes[i].fields[j].field_ref.instance_field.class.internal_class_ref);
	  U1(cap->descriptor_comp.classes[i].fields[j].field_ref.instance_field.token);
	  U2(cap->descriptor_comp.classes[i].fields[j].type.primitive_type);
	}
      }
      if (cap->descriptor_comp.classes[i].method_count > 0) {
	cap->descriptor_comp.classes[i].methods = calloc(cap->descriptor_comp.classes[i].method_count, sizeof(struct method_descriptor_info));
	for (j = 0; j < cap->descriptor_comp.classes[i].method_count; j++) {
	  printf("cap->descriptor_comp.classes[i].methods[%i]:\n", j);
	  U1(cap->descriptor_comp.classes[i].methods[j].token);
	  U1(cap->descriptor_comp.classes[i].methods[j].access_flags);
	  U2(cap->descriptor_comp.classes[i].methods[j].method_offset);
	  U2(cap->descriptor_comp.classes[i].methods[j].type_offset);
	  U2(cap->descriptor_comp.classes[i].methods[j].bytecode_count);
	  U2(cap->descriptor_comp.classes[i].methods[j].exception_handler_count);
	  U2(cap->descriptor_comp.classes[i].methods[j].exception_handler_index);
	}
      }
    }
    U2(cap->descriptor_comp.types.constant_pool_count);
    if (cap->descriptor_comp.types.constant_pool_count > 0) {
      cap->descriptor_comp.types.constant_pool_types = calloc(cap->descriptor_comp.types.constant_pool_count, sizeof(u2));
      for (i = 0 ; i < cap->descriptor_comp.types.constant_pool_count; i++) {
	U2(cap->descriptor_comp.types.constant_pool_types[i]);
      }
    }
    cap->descriptor_comp.types.type_desc_count = 0;
    for (i = 0 ; *index < index_max; i++) {
       if ((cap->descriptor_comp.types.type_desc = realloc(cap->descriptor_comp.types.type_desc, (cap->descriptor_comp.types.type_desc_count + 1) * sizeof(struct struct_type_desc))) == NULL) {
	 JARDIN_FATAL("failed to realloc struct struct_type_desc");
       }
       memset(&cap->descriptor_comp.types.type_desc[cap->descriptor_comp.types.type_desc_count], 0,sizeof(struct struct_type_desc));
       U1(cap->descriptor_comp.types.type_desc[i].nibble_count);
       cap->descriptor_comp.types.type_desc[i].type = calloc((cap->descriptor_comp.types.type_desc[i].nibble_count+1)/2, sizeof(u1));
       for (j = 0; j <  (cap->descriptor_comp.types.type_desc[i].nibble_count+1)/2; j++) {
	 U1(cap->descriptor_comp.types.type_desc[i].type[j]);
       }
       cap->descriptor_comp.types.type_desc_count++;
    }
    break;
    
  }
  return 0;
}


static int jardin_cap_read_file(struct jardin_cap *cap, char *filename) {
  int cap_file;
  struct stat cap_file_stat;
  int ret;
  unsigned char *cap_file_mmap;
  off_t index = 0;
  off_t index_max;
  
  if ((cap_file = open(filename, O_RDONLY)) == -1) {
    JARDIN_ERROR("open of cap file %s failed (%s)", filename, strerror(errno));
    return errno;
  }

  if ((ret = fstat(cap_file, &cap_file_stat)) == -1) {
    JARDIN_ERROR("fstat of file %s failed (%s)", filename, strerror(errno));
    return errno;
  }
  
  JARDIN_DEBUG("Cap file size: 0x%.04lx", cap_file_stat.st_size);
  
  if ((cap_file_mmap = mmap(NULL, cap_file_stat.st_size, PROT_READ, MAP_SHARED, cap_file, 0)) == MAP_FAILED) {
    JARDIN_ERROR("mmap of file %s failed (%s)", filename, strerror(errno));
    return errno;
  }
  
  if (cap_file_mmap[0] < 1 || cap_file_mmap[0] > 12) {
    JARDIN_FATAL("Unknown Component Type (0x%.2x)",cap_file_mmap[0]);
  }
  
  JARDIN_DEBUG("Component Type: %s", jardin_component_types[cap_file_mmap[0]]);
  JARDIN_DEBUG("          size: 0x%.04x", (cap_file_mmap[1] << 8) | cap_file_mmap[2]);
  
  index_max = cap_file_stat.st_size;
  index = 0;
  
  if ( (ret = jardin_cap_fill(cap, cap_file_mmap, &index, index_max)) != 0) {
    JARDIN_FATAL("jardin_cap_fill failed at index 0x%.04lx", index);
  }
  
  // XXXX compare index and index_max
  munmap(cap_file_mmap, cap_file_stat.st_size);
  close(cap_file);
  
  return 0;
}


void jardin_cap_read_files(struct jardin_cap **cap, char *path) {
  unsigned int i;
  char filename[PATH_MAX];

  if ((*cap = calloc(1, sizeof(struct jardin_cap))) == NULL) {
    JARDIN_FATAL("failed to calloc struct jardin_cap");
  }

  method_count = 0; // XXXXX get rid of static method_count
  for (i = 0; jardin_cap_files[i] != NULL; i++ ) {
    snprintf(filename, PATH_MAX, "%s/%s", path, jardin_cap_files[i]);
    JARDIN_DEBUG("filename: %s", filename);
    jardin_cap_read_file(*cap, filename);
  }

}


#define CHECK_AND_FREE(a) if(a!=NULL){free(a);a=NULL;}
void jardin_cap_free(struct jardin_cap **cap) {
  unsigned int i, j;
  
  if (*cap != NULL) {
    /* COMPONENT_Header */
    CHECK_AND_FREE((*cap)->header_comp.this_package.AID);

    /* COMPONRNT_Directory */
    for (i = 0; i < (*cap)->directory_comp.custom_count; i++) { 
      CHECK_AND_FREE((*cap)->directory_comp.custom_components[i].AID);
    }
    CHECK_AND_FREE((*cap)->directory_comp.custom_components);
    
    /* COMPONENT_Applet */
    for (i = 0; i < (*cap)->applet_comp.count; i++) {
      CHECK_AND_FREE((*cap)->applet_comp.applets[i].AID);
    }
    CHECK_AND_FREE((*cap)->applet_comp.applets);

    /* COMPONENT_Import */
    for (i = 0; i < (*cap)->import_comp.count; i++) {
      CHECK_AND_FREE((*cap)->import_comp.packages[i].AID);
    }
    CHECK_AND_FREE((*cap)->import_comp.packages);
    
    /* COMPONENT_ConstantPool */
    CHECK_AND_FREE((*cap)->constant_pool_comp.constant_pool);

    /* COMPONENT_Class */
    for (i = 0; i < (*cap)->class_comp.interface_count; i++) {
      CHECK_AND_FREE((*cap)->class_comp.interfaces[i].superinterfaces);
    }
    CHECK_AND_FREE((*cap)->class_comp.interfaces);
    for (i = 0; i < (*cap)->class_comp.class_count; i++) {
      CHECK_AND_FREE((*cap)->class_comp.classes[i].public_virtual_method_table);
      CHECK_AND_FREE((*cap)->class_comp.classes[i].package_virtual_method_table);
      for (j = 0; j < CLASS_INFO_INTERFACE_COUNT((*cap)->class_comp.classes[i]); j++) {
	CHECK_AND_FREE((*cap)->class_comp.classes[i].interfaces[j].index);
      }
      CHECK_AND_FREE((*cap)->class_comp.classes[i].interfaces);
    }
    CHECK_AND_FREE((*cap)->class_comp.classes);

    /* COMPONENT_Method */
    CHECK_AND_FREE((*cap)->method_comp.exception_handlers);
    for (i = 0; i < (*cap)->method_comp.method_count; i++) {
      CHECK_AND_FREE((*cap)->method_comp.methods[i].bytecodes);
    }
    CHECK_AND_FREE((*cap)->method_comp.methods);
    
    /* COMPONENT_StaticField */
    for (i = 0; i < (*cap)->static_field_comp.array_init_count; i++) {
      CHECK_AND_FREE((*cap)->static_field_comp.array_init[i].values);
    }
    CHECK_AND_FREE((*cap)->static_field_comp.array_init);
    CHECK_AND_FREE((*cap)->static_field_comp.non_default_values);
    
    /* COMPONENT_ReferenceLocation */
    CHECK_AND_FREE((*cap)->ref_location_comp.offsets_to_byte_indices);
    CHECK_AND_FREE((*cap)->ref_location_comp.offsets_to_byte2_indices);

    /* COMPONENT_Export */
    for (i = 0; i < (*cap)->export_comp.class_count; i++) {
      CHECK_AND_FREE((*cap)->export_comp.class_exports[i].static_field_offsets);
      CHECK_AND_FREE((*cap)->export_comp.class_exports[i].static_method_offsets);
    }
    CHECK_AND_FREE((*cap)->export_comp.class_exports);

    /* COMPONENT_Descriptor */
    for (i = 0; i < (*cap)->descriptor_comp.class_count; i++) {
      CHECK_AND_FREE((*cap)->descriptor_comp.classes[i].interfaces);
      CHECK_AND_FREE((*cap)->descriptor_comp.classes[i].fields);
      CHECK_AND_FREE((*cap)->descriptor_comp.classes[i].methods);
    }
    CHECK_AND_FREE((*cap)->descriptor_comp.classes);
    for (i = 0; i < (*cap)->descriptor_comp.types.type_desc_count; i++) {
      CHECK_AND_FREE((*cap)->descriptor_comp.types.type_desc[i].type);
    }
    CHECK_AND_FREE((*cap)->descriptor_comp.types.type_desc);
    CHECK_AND_FREE((*cap)->descriptor_comp.types.constant_pool_types);

    /* and the cap descriptor itself */
    free(*cap);
    *cap = NULL;
  }
}
